/**
 * Java
 *
 * Copyright 2010-2013 IS2T. All rights reserved
 * IS2T PROPRIETARY/CONFIDENTIAL Use is subject to license terms.
 */
package ej.sni;

/**
 * A pool of reusable resources.
 * A buffer is reserved using {@link #reserve()} and released using {@link #release(Object)}.
 * Pool may have a maximum number of resources.
 */
public abstract class PoolOfReusableResources {

	protected int maxNbresources;
	
	protected Object[] resources;
	protected boolean[] resourcesInUse;
	protected int resourcesPtr;

	/**
	 * Allocate a new pool of resources
	 * 
	 * @param maxNbResources a strictly positive integer giving the maximum number of allocated resources,
	 *            or 0 if an unlimited number of resources is allowed
	 */
	public PoolOfReusableResources(int maxNbResources){
		this.maxNbresources = maxNbResources;
		
		int nbResources = (maxNbResources == 0) ? 10 : maxNbResources;
		resources = new Object[nbResources];
		resourcesInUse = new boolean[nbResources];
		resourcesPtr = -1;
	}

	/**
	 * Reserve a buffer. In case all resources are in use and {@link #maxNbresources} is not reached, a new buffer is allocated.
	 * Otherwise this function blocks until a buffer is available
	 * @return an array
	 */
	public synchronized Object reserve(){
		while(true){
			for(int i=resourcesPtr+1; --i>=0;){
				if(!resourcesInUse[i]){
					resourcesInUse[i] = true;
					return resources[i];
				}
			}

			// here, no available buffer
			if(maxNbresources != 0  && resourcesPtr + 1 == maxNbresources){
				// wait for a ready buffer
				try { wait(); }
				catch (InterruptedException e) {}
				// continue a new loop step to retrieve the available buffer
			}
			else
				break;
		}
		
		Object newBuffer = newResource();
		try{ resources[++resourcesPtr] = newBuffer; }
		catch(ArrayIndexOutOfBoundsException e){
			System.arraycopy(resources, 0, resources = new Object[resourcesPtr + 1], 0, resourcesPtr);
			System.arraycopy(resourcesInUse, 0, resourcesInUse = new boolean[resourcesPtr + 1], 0, resourcesPtr);
			resources[resourcesPtr] = newBuffer;
		}
		resourcesInUse[resourcesPtr] = true;
		return newBuffer;
	}
	

	public synchronized void release(Object buffer){
		for(int i=resourcesPtr+1; --i>=0;){
			if(resources[i] == buffer){
				resourcesInUse[i] = false;
				notify(); // awake a potential thread waiting on an available buffer
				return;
			}
		}
	}
	
	/**
	 * Allocate a new resource
	 */
	protected abstract Object newResource();
	
}
