/*
 * Java
 *
 * Copyright 2019-2020 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.sni;

import java.io.PrintStream;
import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * WARNING: structure known by the VM and the S3.
 */
final public class NativeResource {

	private static final int MAX_DESCRIPTION_SIZE = 256;

	/**
	 * Linked list of {@link NativeResource}.
	 */
	private static NativeResource NativeResourceList; // VM Known

	/**
	 * Next {@link NativeResource} in the {@link NativeResource} linked list.
	 */
	private NativeResource next; // VM Known

	/**
	 * Indirect reference to the object that, when it is garbage collected,
	 * will trigger an auto-close of this resource.<p>
	 * May be null.
	 */
	private AutoCloseWeakRef autocloseRef; // VM Known

	/**
	 * Null if the resource has been reclaimed.
	 */
	private long closeFunction; // VM Known

	/**
	 * Null if the resource has been reclaimed.
	 */
	private long descriptionFunction; // VM Known
	/**
	 * -1 if the resource has been reclaimed.
	 */
	private long id; // VM Known

	static {
		// Touch the static so it is always embedded by the SOAR.
		NativeResourceList = null;
	}

	private NativeResource(){
		// Must do nothing: not called by the VM when created
	}

	public long getId() {
		return id;
	}

	final public String getDescription() {
		byte[] buffer = new byte[MAX_DESCRIPTION_SIZE];
		getDescription(this, buffer);
		return SNI.toJavaString(buffer);
	}

	public static void closeOnGC(long resource, long closeFunction, Object obj){
		if(obj == null){
			throw new NullPointerException();
		}

		NativeResource nativeResource = getNativeResource(resource, closeFunction);
		if(nativeResource == null){
			throw new IllegalArgumentException();
		}
		AutoCloseWeakRef autcloseRef = new AutoCloseWeakRef(obj, nativeResource);
		synchronized (nativeResource) {
			if(nativeResource.autocloseRef != null){
				// Autoclose already set for the given native resource.
				throw new IllegalArgumentException();
			}
			// Keep a link to the weak reference to prevent it from garbage collection.
			// The NativeResource are roots so they are never garbage collected (while
			// they are not closed)
			nativeResource.autocloseRef = autcloseRef;
		}
	}

	public static Object clearCloseOnGC(long resource, long closeFunction){
		NativeResource nativeResource = getNativeResource(resource, closeFunction);
		if(nativeResource == null){
			throw new IllegalArgumentException();
		}

		synchronized (nativeResource) {
			AutoCloseWeakRef autocloseRef = nativeResource.autocloseRef;
			if(autocloseRef != null){
				nativeResource.autocloseRef = null;
				return autocloseRef.get();
			}
			else {
				return null;
			}
		}
	}

	public static Iterable<NativeResource> listRegisteredNativeResources(){
		return new Iterable<NativeResource>() {
			@Override
			public Iterator<NativeResource> iterator() {
				return new NativeResource.NativeResourceIterator();
			}
		};
	}

	public static void printRegisteredNativeResources(PrintStream out){
		byte[] buffer = new byte[MAX_DESCRIPTION_SIZE];
		try{
			out.println("Id    \tOwner  \tDescription");
			for(NativeResource resource : listRegisteredNativeResources()){
				out.print(resource.getId());
				out.print('\t');
				out.print(resource.getOwner());
				out.print('\t');
				getDescription(resource, buffer);
				out.println(SNI.toJavaString(buffer));
			}
		}
		catch(Throwable t){
			// Sanity check
			out.print("ERROR");
		}
	}

	static class NativeResourceIterator implements Iterator<NativeResource>{

		private NativeResource next;

		public NativeResourceIterator(){
			this.next = NativeResourceList;
		}

		@Override
		public boolean hasNext() {
			return this.next != null;
		}

		@Override
		public NativeResource next() {
			NativeResource result = this.next;
			if(result != null){
				this.next = result.next;
				return result;
			}
			else {
				throw new NoSuchElementException();
			}
		}

		@Override
		public void remove() {
			throw new UnsupportedOperationException();
		}

	}

	/**
	 * Returns the number of registered native resources.
	 */
	native public static int getRegisteredNativeResourcesCount();

	/**
	 * Fills the given buffer with a null terminated string.
	 */
	native private static void getDescription(NativeResource nativeResource, byte[] buffer);

	/**
	 * Returns the native resource registered with the given <code>resource</code> and <code>closeFunction</code> pair.
	 * Returns null if the native resource is not found.
	 * <p>
	 * @param resource the <code>resource</code> argument given to <code>SNI_registerResource()</code> when the native resource has been registered.
	 * @param closeFunction the <code>close</code> argument given to <code>SNI_registerResource()</code> when the native resource has been registered.
	 * 
	 * @return a {@link NativeResource} or null if no {@link NativeResource} has been registered for the given <code>resource</code> and <code>closeFunction</code> pair.
	 */
	native public static NativeResource getNativeResource(long resource, long closeFunction);

	native public Object getOwner();
}
