/*
 * Java
 *
 * Copyright 2020-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg.image;

import ej.microui.display.GraphicsContext;
import ej.microvg.Matrix;
import ej.microvg.VectorImage;

/**
 * Represents an element of a {@link VectorImage}: a group or a path associated with a color or a gradient.
 */
public interface ImageElement {

	/**
	 * Draws the image element to a graphics context.
	 *
	 * @param g
	 *            the graphics context
	 * @param matrix
	 *            the matrix to apply
	 */
	void draw(GraphicsContext g, Matrix matrix);

	/**
	 * Draws the image element to a graphics context.
	 *
	 * @param g
	 *            the graphics context
	 * @param matrix
	 *            the matrix to apply
	 * @param alpha
	 *            the global opacity rendering value
	 */
	void draw(GraphicsContext g, Matrix matrix, int alpha);

	/**
	 * Draws the image element to a graphics context at specific time.
	 *
	 * @param g
	 *            the graphics context
	 * @param matrix
	 *            the matrix to apply
	 * @param elapsedTime
	 *            the time elapsed within the overall animation, in milliseconds
	 */
	void drawAnimated(GraphicsContext g, Matrix matrix, long elapsedTime);

	/**
	 * Draws the image element to a graphics context at specific time.
	 *
	 * @param g
	 *            the graphics context
	 * @param matrix
	 *            the matrix to apply
	 * @param alpha
	 *            the global opacity rendering value
	 * @param elapsedTime
	 *            the time elapsed within the overall animation, in milliseconds
	 */
	void drawAnimated(GraphicsContext g, Matrix matrix, int alpha, long elapsedTime);

	/**
	 * Draws the image element to a graphics context after transforming it.
	 *
	 * @param g
	 *            the graphics context
	 * @param matrix
	 *            the matrix to apply
	 * @param transformer
	 *            the transformer to apply
	 */
	void drawTransformed(GraphicsContext g, Matrix matrix, ImageElementTransformer transformer);

	/**
	 * Draws the image element to a graphics context after transforming it at specific time.
	 *
	 * @param g
	 *            the graphics context
	 * @param matrix
	 *            the matrix to apply
	 * @param elapsedTime
	 *            the time elapsed within the overall animation, in milliseconds
	 * @param transformer
	 *            the transformer to apply
	 */
	void drawTransformedAnimated(GraphicsContext g, Matrix matrix, long elapsedTime,
			ImageElementTransformer transformer);

	/**
	 * Transforms the image element with the given transformer.
	 *
	 * @param transformer
	 *            the transformer to apply
	 * @return a transformed copy of the image element
	 */
	ImageElement transform(ImageElementTransformer transformer);
}
