/*
 * Java
 *
 * Copyright 2020-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg.image;

import ej.microui.display.GraphicsContext;
import ej.microvg.Matrix;
import ej.microvg.VectorImage;

/**
 * Represents an {@link VectorImage} clip.
 */
public class ImageClip {

	private final int initialX;
	private final int initialY;
	private final int initialWidth;
	private final int initialHeight;

	/**
	 * Creates an ImageClip from graphic context.
	 *
	 * @param g
	 *            graphic context to get the initial clipping area from
	 *
	 */
	public ImageClip(GraphicsContext g) {
		// Save initial clipping values
		this.initialX = g.getClipX();
		this.initialY = g.getClipY();
		this.initialWidth = g.getClipWidth();
		this.initialHeight = g.getClipHeight();
	}

	/**
	 * Applies the clip to a graphic context.
	 *
	 * @param g
	 *            graphic context on which the clip is applied
	 * @param matrix
	 *            transformation matrix
	 * @param width
	 *            width of image viewport
	 * @param height
	 *            height of image viewport
	 */
	public static void apply(GraphicsContext g, Matrix matrix, int width, int height) {
		// Transform each corners points of the image viewbox
		// top/left, not impacted by scale and rotate
		int x0 = (int) matrix.values[2];
		int y0 = (int) matrix.values[5];

		// top/right
		int x1 = (int) (width * matrix.values[0] + matrix.values[2]);
		int y1 = (int) (width * matrix.values[3] + matrix.values[5]);

		// bottom/Left
		int x2 = (int) (height * matrix.values[1] + matrix.values[2]);
		int y2 = (int) (height * matrix.values[4] + matrix.values[5]);

		// bottom/right
		int x3 = (int) (width * matrix.values[0] + height * matrix.values[1] + matrix.values[2]);
		int y3 = (int) (width * matrix.values[3] + height * matrix.values[4] + matrix.values[5]);

		// Compute the clipping area from all points
		int xmin = Math.min(x0, Math.min(x1, Math.min(x2, x3)));
		int ymin = Math.min(y0, Math.min(y1, Math.min(y2, y3)));
		int xmax = Math.max(x0, Math.max(x1, Math.max(x2, x3)));
		int ymax = Math.max(y0, Math.max(y1, Math.max(y2, y3)));

		// apply the clipping area to graphic context
		g.intersectClip(xmin, ymin, xmax - xmin, ymax - ymin);
	}

	/**
	 * Reverts the initial clip to initial clipping area.
	 *
	 * @param g
	 *            graphic context on which the clip is applied
	 */
	public void revert(GraphicsContext g) {
		g.setClip(this.initialX, this.initialY, this.initialWidth, this.initialHeight);
	}

}
