/*
 * Java
 *
 * Copyright 2020-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg.image;

import ej.microui.display.GraphicsContext;
import ej.microvg.Matrix;
import ej.microvg.VectorImage;

/**
 * Represents an element of a {@link VectorImage}: a group of paths or groups.
 */
public class GroupElement implements ImageElement {

	final ImageElement[] children;

	/**
	 * Cannot create the object at runtime: it is always an immutable object. However the constructor is required to
	 * prevent Java compiler errors.
	 */
	/* package */ GroupElement(ImageElement[] children) {
		this.children = children;
	}

	/* package */ GroupElement(GroupElement element, ImageElementTransformer transformer) {
		int length = element.children.length;
		this.children = new ImageElement[length];
		for (int i = 0; i < length; i++) {
			this.children[i] = element.children[i].transform(transformer);
		}
	}

	// Draw
	@Override
	public void draw(GraphicsContext g, Matrix matrix) {
		Matrix localMatrix = updateMatrix(matrix, 0, false);

		for (ImageElement child : this.children) {
			child.draw(g, localMatrix);
		}
	}

	@Override
	public void draw(GraphicsContext g, Matrix matrix, int alpha) {
		Matrix localMatrix = updateMatrix(matrix, 0, false);
		for (ImageElement child : this.children) {
			child.draw(g, localMatrix, alpha);
		}
	}

	@Override
	public void drawAnimated(GraphicsContext g, Matrix matrix, long elapsedTime) {
		Matrix localMatrix = updateMatrix(matrix, elapsedTime, true);

		for (ImageElement child : this.children) {
			child.drawAnimated(g, localMatrix, elapsedTime);
		}
	}

	@Override
	public void drawAnimated(GraphicsContext g, Matrix matrix, int alpha, long elapsedTime) {
		Matrix localMatrix = updateMatrix(matrix, elapsedTime, true);

		for (ImageElement child : this.children) {
			child.drawAnimated(g, localMatrix, alpha, elapsedTime);
		}
	}

	@Override
	public void drawTransformed(GraphicsContext g, Matrix matrix, ImageElementTransformer transformer) {
		Matrix localMatrix = updateMatrix(matrix, 0, false);
		for (ImageElement child : this.children) {
			child.drawTransformed(g, localMatrix, transformer);
		}
	}

	@Override
	public void drawTransformedAnimated(GraphicsContext g, Matrix matrix, long elapsedTime,
			ImageElementTransformer transformer) {
		Matrix localMatrix = updateMatrix(matrix, elapsedTime, true);

		for (ImageElement child : this.children) {
			child.drawTransformedAnimated(g, localMatrix, elapsedTime, transformer);
		}
	}

	/**
	 * updates the given matrix with animations or transformations if any.
	 *
	 * @param globalMatrix
	 *            the matrix global matrix used as updated matrix starting transformation
	 * @param atTime
	 *            the time elapsed within the overall animation, in milliseconds
	 * @param applyAnimations
	 *            if true, apply animations otherwise only transformations
	 * @return the updated matrix
	 *
	 */
	/* package */ Matrix updateMatrix(Matrix globalMatrix, long atTime, boolean applyAnimations) {
		// No animation, return initial gradient
		return globalMatrix;
	}

	@Override
	public ImageElement transform(ImageElementTransformer transformer) {
		return new GroupElement(this, transformer);
	}

}
