/*
 * Java
 *
 * Copyright 2020-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg.image;

import ej.microui.display.GraphicsContext;
import ej.microvg.BlendMode;
import ej.microvg.ColorHelper;
import ej.microvg.Matrix;
import ej.microvg.PainterNatives;
import ej.microvg.VectorGraphicsException;
import ej.microvg.VectorGraphicsNatives;
import ej.microvg.VectorImage;

/**
 * Represents an element of a {@link VectorImage}: a path associated with a gradient.
 */
public class GradientElement extends PathElement {

	private static final int BASIC_ENCODER_HEADER = 7;
	private static final int STOPS_COUNT_IDX = 0;

	/* package */ final int[] gradient;

	/**
	 * Creates a gradient element from a path data, a fill rule and a gradient.
	 *
	 * @param path
	 *            the path data
	 * @param fillRule
	 *            the fill rule to use when drawing the path
	 * @param gradient
	 *            the gradient to use when filling the path
	 */
	public GradientElement(byte[] path, int fillRule, int[] gradient) {
		super(path, fillRule);
		this.gradient = gradient;
	}

	/* package */ GradientElement(GradientElement element, ImageElementTransformer transformer) {
		super(element.path, element.fillRule);
		this.gradient = applyTransform(element.gradient, transformer);
	}

	@Override
	public void draw(GraphicsContext g, Matrix matrix) {
		draw(g, this.path, matrix, this.gradient);
	}

	@Override
	public void draw(GraphicsContext g, Matrix matrix, int alpha) {
		int[] gradientWithAlpha = applyAlpha(this.gradient, alpha);
		draw(g, this.path, matrix, gradientWithAlpha);
	}

	@Override
	public void drawAnimated(GraphicsContext g, Matrix matrix, long atTime) {
		draw(g, getPathAtTime(atTime), matrix, getGradientAtTime(this.gradient, atTime));
	}

	@Override
	public void drawAnimated(GraphicsContext g, Matrix matrix, int alpha, long atTime) {
		int[] gradientWithAlpha = applyAlpha(this.gradient, alpha);
		draw(g, getPathAtTime(atTime), matrix, getGradientAtTime(gradientWithAlpha, atTime));
	}

	@Override
	public void drawTransformed(GraphicsContext g, Matrix matrix, ImageElementTransformer transformer) {
		int[] transformedGradient = applyTransform(this.gradient, transformer);
		draw(g, this.path, matrix, transformedGradient);
	}

	@Override
	public void drawTransformedAnimated(GraphicsContext g, Matrix matrix, long atTime,
			ImageElementTransformer transformer) {
		int[] transformedGradient = applyTransform(this.gradient, transformer);
		draw(g, getPathAtTime(atTime), matrix, getGradientAtTime(transformedGradient, atTime));
	}

	private void draw(GraphicsContext g, byte[] path, Matrix matrix, int[] gradient) {
		if (VectorGraphicsNatives.RET_SUCCESS != PainterNatives.drawGradient(g.getSNIContext(), path,
				g.getTranslationX(), g.getTranslationY(), matrix.values, this.fillRule, GraphicsContext.OPAQUE,
				BlendMode.SRC_OVER.ordinal(), gradient, new Matrix().getSNIContext())) {
			throw new VectorGraphicsException(VectorGraphicsException.IMAGE_INVALID);
		}
	}

	/* package */ int[] applyAlpha(int[] gradient, int alpha) {
		int[] newGradient = gradient.clone();

		// Transform colors
		for (int i = 0; i < gradient[STOPS_COUNT_IDX]; i++) {
			int color = gradient[BASIC_ENCODER_HEADER + i];
			int colorAlpha = ColorHelper.getAlpha(color);
			int resultAlpha = colorAlpha * alpha / ColorHelper.MAX_COLOR_COMPONENT_VALUE;
			newGradient[BASIC_ENCODER_HEADER + i] = ColorHelper.updateAlpha(color, (byte) resultAlpha);
		}
		return newGradient;
	}

	private int[] applyTransform(int[] gradient, ImageElementTransformer transformer) {
		int[] newGradient = gradient.clone();

		// Transform colors
		for (int i = 0; i < gradient[STOPS_COUNT_IDX]; i++) {
			int color = gradient[BASIC_ENCODER_HEADER + i];
			newGradient[BASIC_ENCODER_HEADER + i] = transformer.transformColor(color);
		}
		return newGradient;
	}

	/* package */ byte[] getPathAtTime(long atTime) {
		return this.path;
	}

	/* package */ int[] getGradientAtTime(int[] gradient, long atTime) {
		return gradient;
	}

	@Override
	public PathElement transform(ImageElementTransformer transformer) {
		return new GradientElement(this, transformer);
	}

}
