/*
 * Copyright 2022-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg.image;

import ej.bon.XMath;
import ej.microui.display.GraphicsContext;
import ej.microvg.ColorHelper;

/**
 * Transformer that changes colors using a matrix 4x5.
 */
public class ColorMatrixTransformer implements ImageElementTransformer {

	private static final int COLOR_MATRIX_WIDTH = 5;
	private static final int RED_OFFSET = 0;
	private static final int GREEN_OFFSET = 1;
	private static final int BLUE_OFFSET = 2;
	private static final int ALPHA_OFFSET = 3;
	private static final int CONSTANT_OFFSET = 4;
	private static final int RED_LINE = COLOR_MATRIX_WIDTH * RED_OFFSET;
	private static final int GREEN_LINE = COLOR_MATRIX_WIDTH * GREEN_OFFSET;
	private static final int BLUE_LINE = COLOR_MATRIX_WIDTH * BLUE_OFFSET;
	private static final int ALPHA_LINE = COLOR_MATRIX_WIDTH * ALPHA_OFFSET;

	private final float[] colorMatrix;

	/**
	 * Creates a color matrix transformer.
	 *
	 * @param colorMatrix
	 *            the color matrix
	 */
	public ColorMatrixTransformer(float[] colorMatrix) {
		this.colorMatrix = colorMatrix;
	}

	@Override
	public int transformColor(int color) {
		int red = ColorHelper.getRed(color);
		int green = ColorHelper.getGreen(color);
		int blue = ColorHelper.getBlue(color);
		int alpha = ColorHelper.getAlpha(color);

		float[] colorMatrix = this.colorMatrix;
		int resultRed = computeComponent(red, green, blue, alpha, colorMatrix, RED_LINE);
		int resultGreen = computeComponent(red, green, blue, alpha, colorMatrix, GREEN_LINE);
		int resultBlue = computeComponent(red, green, blue, alpha, colorMatrix, BLUE_LINE);
		int resultAlpha = computeComponent(red, green, blue, alpha, colorMatrix, ALPHA_LINE);
		return ColorHelper.getColor(resultAlpha, resultRed, resultGreen, resultBlue);
	}

	private int computeComponent(int red, int green, int blue, int alpha, float[] colorMatrix, int startIndex) {
		float redFactor = colorMatrix[startIndex + RED_OFFSET] * red;
		float greenFactor = colorMatrix[startIndex + GREEN_OFFSET] * green;
		float blueFactor = colorMatrix[startIndex + BLUE_OFFSET] * blue;
		float alphaFactor = colorMatrix[startIndex + ALPHA_OFFSET] * alpha;
		float hardcodedColor = colorMatrix[startIndex + CONSTANT_OFFSET];
		return XMath.limit((int) (redFactor + greenFactor + blueFactor + alphaFactor + hardcodedColor), 0x0, 0xff);
	}

	/**
	 * Checks whether a color matrix may turn a color as transparent.
	 * <p>
	 * NB: this check only works if the original color is opaque.
	 *
	 * @param colorMatrix
	 *            the color matrix
	 * @return <code>true</code> if the color matrix keeps an opaque color, false otherwise.
	 */
	public static boolean keepsOpaque(float[] colorMatrix) {
		return (colorMatrix[ALPHA_LINE + ALPHA_OFFSET] == 1f
				|| colorMatrix[ALPHA_LINE + CONSTANT_OFFSET] == GraphicsContext.OPAQUE)
				&& colorMatrix[ALPHA_LINE + RED_OFFSET] == 0 && colorMatrix[ALPHA_LINE + GREEN_OFFSET] == 0
				&& colorMatrix[ALPHA_LINE + BLUE_OFFSET] == 0;
	}

}
