/*
 * Java
 *
 * Copyright 2020-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg.image;

import ej.microui.display.GraphicsContext;
import ej.microvg.BlendMode;
import ej.microvg.ColorHelper;
import ej.microvg.Matrix;
import ej.microvg.PainterNatives;
import ej.microvg.VectorGraphicsException;
import ej.microvg.VectorGraphicsNatives;
import ej.microvg.VectorImage;

/**
 * Represents an element of a {@link VectorImage}: a path associated with a color.
 */
public class ColorElement extends PathElement {

	/** The color filling for this element. */
	protected final int color;

	/**
	 * Creates a color element from a path data, a fill rule and a color.
	 *
	 * @param path
	 *            the path data
	 * @param fillRule
	 *            the fill rule to use when drawing the path
	 * @param color
	 *            the color to use when filling the path
	 */
	public ColorElement(byte[] path, int fillRule, int color) {
		super(path, fillRule);
		this.color = color;
	}

	/* package */ ColorElement(ColorElement element, ImageElementTransformer transformer) {
		super(element.path, element.fillRule);
		this.color = transformer.transformColor(element.color);
	}

	@Override
	public void draw(GraphicsContext g, Matrix matrix) {
		drawWithColor(g, this.path, matrix, this.color);
	}

	@Override
	public void draw(GraphicsContext g, Matrix matrix, int alpha) {
		int colorWithAlpha = applyAlpha(this.color, alpha);
		drawWithColor(g, this.path, matrix, colorWithAlpha);
	}

	@Override
	public void drawAnimated(GraphicsContext g, Matrix matrix, long atTime) {
		int colorAtTime = getColorAtTime(atTime);
		drawWithColor(g, getPathAtTime(atTime), matrix, colorAtTime);
	}

	@Override
	public void drawAnimated(GraphicsContext g, Matrix matrix, int alpha, long atTime) {
		int colorAtTime = getColorAtTime(atTime);
		int colorWithAlpha = applyAlpha(colorAtTime, alpha);
		drawWithColor(g, getPathAtTime(atTime), matrix, colorWithAlpha);
	}

	@Override
	public void drawTransformed(GraphicsContext g, Matrix matrix, ImageElementTransformer transformer) {
		int transformedColor = transformer.transformColor(this.color);
		drawWithColor(g, this.path, matrix, transformedColor);
	}

	@Override
	public void drawTransformedAnimated(GraphicsContext g, Matrix matrix, long atTime,
			ImageElementTransformer transformer) {
		int colorAtTime = getColorAtTime(atTime);
		int transformedColor = transformer.transformColor(colorAtTime);
		drawWithColor(g, getPathAtTime(atTime), matrix, transformedColor);
	}

	private void drawWithColor(GraphicsContext g, byte[] path, Matrix matrix, int color) {
		if (VectorGraphicsNatives.RET_SUCCESS != PainterNatives.drawPath(g.getSNIContext(), path, g.getTranslationX(),
				g.getTranslationY(), matrix.values, this.fillRule, BlendMode.SRC_OVER.ordinal(), color)) {
			throw new VectorGraphicsException(VectorGraphicsException.IMAGE_INVALID);
		}
	}

	private int applyAlpha(int color, int alpha) {
		int colorAlpha = ColorHelper.getAlpha(color);
		int resultAlpha = colorAlpha * alpha / ColorHelper.MAX_COLOR_COMPONENT_VALUE;
		return ColorHelper.updateAlpha(color, (byte) resultAlpha);
	}

	/* package */ int getColorAtTime(long atTime) {
		// No animation, return initial color
		return this.color;
	}

	/* package */ byte[] getPathAtTime(long atTime) {
		// No animation, return initial path
		return this.path;
	}

	@Override
	public PathElement transform(ImageElementTransformer transformer) {
		return new ColorElement(this, transformer);
	}

}
