/*
 * Java
 *
 * Copyright 2022-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg;

import ej.microui.display.GraphicsContext;
import ej.microvg.VectorGraphicsPainter.FillType;
import ej.microvg.image.ColorElement;
import ej.microvg.image.GradientElement;
import ej.microvg.image.ImageElement;

/**
 * A builder for vector images.
 *
 * <p>
 * Users can add the paths that compose the image and build the corresponding {@link VectorImage} instance. The paths
 * will be drawn in the order of addition.
 */
public class VectorImageBuilder {

	private static final int MAX_TRANSPARENT_COLOR = 0x00FFFFFF;

	private static final int MIN_TRANSPARENT_COLOR = 0x00000000;

	private ImageElement[] elements;

	private final float height;

	private final float width;

	/**
	 * Creates a new builder for vector images.
	 *
	 * @param width
	 *            the width of the image
	 * @param height
	 *            the height of the image
	 * @throws IllegalArgumentException
	 *             if either <code>width</code> or <code>height</code> is negative or zero.
	 */
	public VectorImageBuilder(float width, float height) {
		if (width <= 0 || height <= 0) {
			throw new IllegalArgumentException();
		}
		this.width = width;
		this.height = height;
		this.elements = new ImageElement[0];
	}

	/**
	 * Adds the given path with specified color and fill type.
	 *
	 * <p>
	 * The given color value is interpreted as a 32-bit ARGB color, where the high-order byte is the alpha channel and
	 * the remaining bytes contain the red, green and blue channels, respectively. A fully opaque color will have an
	 * alpha of {@code 255}, while a fully transparent color will have an alpha of {@code 0}. For example, passing the
	 * color {@code 0xff0000ff} to this method results in a path with an opaque blue fill.
	 *
	 * <p>
	 * If the specified color is fully transparent, the path is not added.
	 *
	 * @param path
	 *            the path to add
	 * @param color
	 *            the color to fill the path with
	 * @param fillType
	 *            the fill type to use
	 * @return the builder instance.
	 * @see GraphicsContext#OPAQUE
	 * @see GraphicsContext#TRANSPARENT
	 */
	public VectorImageBuilder addPath(Path path, int color, FillType fillType) {
		if (!path.isEmpty() && checkColor(color)) {
			addElement(new ColorElement(path.getData(), fillType.ordinal(), color));
		}
		return this;
	}

	private boolean checkColor(int color) {
		return color < MIN_TRANSPARENT_COLOR || color > MAX_TRANSPARENT_COLOR;
	}

	/**
	 * Adds the given path with specified gradient and fill type.
	 *
	 * <p>
	 * The colors of the gradient are interpreted as 32-bit ARGB colors, where the high-order byte is the alpha channel
	 * and the remaining bytes contain the red, green and blue channels, respectively. A fully opaque color will have an
	 * alpha of {@code 255}, while a fully transparent color will have an alpha of {@code 0}. For example, the color
	 * {@code 0xff0000ff} results in an opaque blue.
	 *
	 * @param path
	 *            the path to add
	 * @param gradient
	 *            the gradient to fill the path with
	 * @param fillType
	 *            the fill type to use
	 * @return the builder instance.
	 * @see GraphicsContext#OPAQUE
	 * @see GraphicsContext#TRANSPARENT
	 */
	public VectorImageBuilder addGradientPath(Path path, LinearGradient gradient, FillType fillType) {
		if (!path.isEmpty()) {
			addElement(new GradientElement(path.getData(), fillType.ordinal(), gradient.sniContext));
		}
		return this;
	}

	/**
	 * Creates the vector image from the builder state.
	 * <p>
	 * The returned image is allocated dynamically and must be closed explicitly.
	 *
	 * @return a vector image
	 */
	public ResourceVectorImage build() {
		return new InternalVectorImage(this.width, this.height, this.elements);
	}

	private void addElement(ImageElement element) {
		ImageElement[] source = this.elements;
		int oldLength = source.length;
		int newLength = oldLength + 1;
		ImageElement[] destination = new ImageElement[newLength];
		System.arraycopy(source, 0, destination, 0, oldLength);
		destination[oldLength] = element;
		this.elements = destination;
	}
}
