/*
 * Java
 *
 * Copyright 2021-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg;

import ej.bon.XMath;
import ej.microui.display.GraphicsContext;
import ej.microvg.image.ColorMatrixTransformer;
import ej.sni.SNI;

/**
 * Represents a vector image.
 */
public class VectorImage {

	static {
		VectorGraphicsNatives.startup();
	}

	/**
	 * Raw image metadata: used to retrieve image characteristics when opening an image.
	 */
	private static final int RAW_OFFSET_F32_WIDTH = 0;
	private static final int RAW_OFFSET_F32_HEIGHT = 1;
	private static final int RAW_OFFSET_U32_DURATION = 2;
	protected static final int RAW_OFFSET_U32_FLAGS = 3;
	protected static final int RAW_METADATA_SIZE = RAW_OFFSET_U32_FLAGS + 1;

	/**
	 * Raw image metadata: flags masks
	 *
	 * Subclasses define extra flags. Check in subclasses when defining flags to make sure values do not overlap.
	 */
	private static final int RAW_FLAG_OVERLAP_PATH = 0x01;

	/**
	 * Raw image SNI context: as soon as an image is opened, use this context to target the image in the native world.
	 */
	private static final int SNICONTEXT_SIZE = 8;

	/**
	 * Elapses time to set to draw the image without animation.
	 */
	private static final int DRAW_WITHOUT_ANIMATION = -1;

	private final byte[] sniContext;
	private final float width;
	private final float height;
	private final long duration;
	private final boolean hasOverlappingElements;

	/* package */ VectorImage(byte[] sniContext, int[] metadata) {

		this.sniContext = sniContext;

		// retrieve image data from metadata
		this.width = Float.intBitsToFloat(metadata[RAW_OFFSET_F32_WIDTH]);
		this.height = Float.intBitsToFloat(metadata[RAW_OFFSET_F32_HEIGHT]);
		this.duration = metadata[RAW_OFFSET_U32_DURATION];

		int flags = metadata[RAW_OFFSET_U32_FLAGS];
		this.hasOverlappingElements = (flags & RAW_FLAG_OVERLAP_PATH) == RAW_FLAG_OVERLAP_PATH;
	}

	/* package */ VectorImage(VectorImage source, byte[] sniContext) {

		this.sniContext = sniContext;

		// retrieve image data from source
		this.width = source.width;
		this.height = source.height;
		this.duration = source.duration;
		this.hasOverlappingElements = source.hasOverlappingElements;
	}

	/* package */ VectorImage(float width, float height) {
		this.width = width;
		this.height = height;
		this.duration = 0;
		this.hasOverlappingElements = false;
		this.sniContext = createSNIContext();
	}

	protected static final byte[] resourcePathToSniPath(String resourcePath) {
		if (resourcePath.length() == 0 || resourcePath.charAt(0) != '/') {
			throw new VectorGraphicsException(VectorGraphicsException.IMAGE_INVALID_PATH);
		}

		byte[] sniPath = SNI.toCString(resourcePath + "_raw");
		if (null == sniPath) {
			// cannot be here but have to check the @NonNull rule
			throw new RuntimeException();
		}

		return sniPath;
	}

	/**
	 * Gets a vector image from a path.
	 *
	 * @param resourcePath
	 *            the path to get the image from
	 * @return a vector image
	 * @throws VectorGraphicsException
	 *             if the image could not be retrieved for any reason (see
	 *             {@link VectorGraphicsException#getErrorCode()})
	 */
	public static VectorImage getImage(String resourcePath) {
		byte[] sniPath = resourcePathToSniPath(resourcePath);
		byte[] sniContext = createSNIContext();
		int[] metadata = new int[RAW_METADATA_SIZE];

		int err = VectorGraphicsNatives.getImage(sniPath, sniPath.length, sniContext, metadata);
		if (0 > err) {
			// the native returns the same error codes than the library
			throw new VectorGraphicsException(err);
		}

		return new VectorImage(sniContext, metadata);
	}

	/**
	 * Creates the context used to identify the image on native side
	 *
	 * @return a byte array whose format is known by native side.
	 */
	/* package */ static byte[] createSNIContext() {
		return new byte[SNICONTEXT_SIZE];
	}

	/**
	 * Gets the width of the vector image.
	 *
	 * @return the width
	 */
	public float getWidth() {
		return this.width;
	}

	/**
	 * Gets the height of the vector image.
	 *
	 * @return the height
	 */
	public float getHeight() {
		return this.height;
	}

	/**
	 * Gets the duration of the vector image animation if the image is animated, otherwise 0.
	 *
	 * @return the duration of the animation of the vector image
	 */
	public long getDuration() {
		return this.duration;
	}

	/**
	 * Creates an image derived from this image, applying the given color matrix.
	 * <p>
	 * The returned image is allocated dynamically and must be closed explicitly.
	 * <p>
	 * The given matrix is a 4x5 color matrix. It is organized like that:
	 * <ul>
	 * <li>Each line is used to compute a component of the resulting color, in this order: red, green, blue, alpha.</li>
	 * <li>The four first columns are multipliers applied to a component of the initial color, in this order: red,
	 * green, blue, alpha.</li>
	 * <li>The last column is a constant value.</li>
	 * </ul>
	 * Each component is then computed like that:
	 * <code>redMultiplier x redInitialComponent + greenMultiplier x greenInitialComponent + blueMultiplier x blueInitialComponent + constant</code>
	 * Each component is clamped between 0x0 and 0xff.
	 * <p>
	 * Let A, R, G, B be the components of the initial color and the following array a color matrix:
	 *
	 * <pre>
	 * <code>
	 * { rR, rG, rB, rA, rC,
	 *   gR, gG, gB, gA, gC,
	 *   bR, bG, bB, bA, bC,
	 *   aR, aG, aB, aA, aC }
	 * </code>
	 * </pre>
	 * <p>
	 * The resulting color components are computed as:
	 *
	 * <pre>
	 * resultRed = rR * R + rG * G + rB * B + rA * A + rC
	 * resultGreen = gR * R + gG * G + gB * B + gA * A + gC
	 * resultBlue = bR * R + bG * G + bB * B + bA * A + bC
	 * resultAlpha = aR * R + aG * G + aB * B + aA * A + aC
	 * </pre>
	 *
	 * @param colorMatrix
	 *            the color matrix used to transform colors
	 * @return the filtered image
	 * @throws ArrayIndexOutOfBoundsException
	 *             if the given color matrix is shorter than 20 entries
	 * @throws VectorGraphicsException
	 *             if the image could not be drawn for any reason (see {@link VectorGraphicsException#getErrorCode()})
	 */
	public ResourceVectorImage filterImage(float[] colorMatrix) {
		checkOverlapAlpha(colorMatrix);

		byte[] sniContext = createSNIContext();
		checkReturnCode(VectorGraphicsNatives.createImage(getSNIContext(), sniContext, colorMatrix));

		// Create a ResourceVectorImage which is closeable (to free the image buffer)
		return new ResourceVectorImage(this, sniContext, true);
	}

	/**
	 * Draws the paths with given matrix.
	 *
	 * @param g
	 *            the graphics context to draw on.
	 * @param matrix
	 *            the matrix to apply
	 */
	/* package */ void draw(GraphicsContext g, Matrix matrix) {
		checkReturnCode(PainterNatives.drawImage(g.getSNIContext(), getSNIContext(), g.getTranslationX(),
				g.getTranslationY(), matrix.getSNIContext(), GraphicsContext.OPAQUE, DRAW_WITHOUT_ANIMATION, null));
	}

	/**
	 * Draws the paths with given matrix.
	 *
	 * @param g
	 *            the graphics context to draw on
	 * @param matrix
	 *            the matrix to apply
	 * @param alpha
	 *            the global opacity rendering value
	 */
	/* package */ void draw(GraphicsContext g, Matrix matrix, int alpha) {
		checkOverlapAlpha(alpha);
		checkReturnCode(PainterNatives.drawImage(g.getSNIContext(), getSNIContext(), g.getTranslationX(),
				g.getTranslationY(), matrix.getSNIContext(), alpha, DRAW_WITHOUT_ANIMATION, null));
	}

	/**
	 * Draws the paths with given matrix at a specific time.
	 *
	 * @param g
	 *            the graphics context to draw on
	 * @param matrix
	 *            the matrix to apply
	 * @param elapsedTime
	 *            the time elapsed within the overall animation, in milliseconds
	 * @throws IllegalArgumentException
	 *             if the elapsed time is negative
	 */
	/* package */ void drawAnimated(GraphicsContext g, Matrix matrix, long elapsedTime) {
		elapsedTime = checkElapsedTime(elapsedTime);
		checkReturnCode(PainterNatives.drawImage(g.getSNIContext(), getSNIContext(), g.getTranslationX(),
				g.getTranslationY(), matrix.getSNIContext(), GraphicsContext.OPAQUE, elapsedTime, null));
	}

	/**
	 * Draws the paths with given matrix at a specific time.
	 *
	 * @param g
	 *            the graphics context to draw on
	 * @param matrix
	 *            the matrix to apply
	 * @param alpha
	 *            the global opacity rendering value
	 * @param elapsedTime
	 *            the time elapsed within the overall animation, in milliseconds
	 * @throws IllegalArgumentException
	 *             if the elapsed time is negative
	 */
	/* package */ void drawAnimated(GraphicsContext g, Matrix matrix, int alpha, long elapsedTime) {
		checkOverlapAlpha(alpha);
		elapsedTime = checkElapsedTime(elapsedTime);
		checkReturnCode(PainterNatives.drawImage(g.getSNIContext(), getSNIContext(), g.getTranslationX(),
				g.getTranslationY(), matrix.getSNIContext(), alpha, elapsedTime, null));
	}

	/**
	 * Draws the paths with given matrix and with colors transformed with the given matrix.
	 *
	 * @param g
	 *            the graphics context to draw on
	 * @param matrix
	 *            the matrix to apply
	 * @param colorMatrix
	 *            the color matrix to apply
	 */
	/* package */ void drawFiltered(GraphicsContext g, Matrix matrix, float[] colorMatrix) {
		checkOverlapAlpha(colorMatrix);
		checkReturnCode(
				PainterNatives.drawImage(g.getSNIContext(), getSNIContext(), g.getTranslationX(), g.getTranslationY(),
						matrix.getSNIContext(), GraphicsContext.OPAQUE, DRAW_WITHOUT_ANIMATION, colorMatrix));
	}

	/**
	 * Draws the paths with given matrix at a specific time and with colors transformed with the given matrix.
	 *
	 * @param g
	 *            the graphics context to draw on
	 * @param matrix
	 *            the matrix to apply
	 * @param colorMatrix
	 *            the color matrix to apply
	 * @param elapsedTime
	 *            the time elapsed within the overall animation, in milliseconds
	 */
	/* package */ void drawFilteredAnimated(GraphicsContext g, Matrix matrix, long elapsedTime, float[] colorMatrix) {
		checkOverlapAlpha(colorMatrix);
		checkReturnCode(PainterNatives.drawImage(g.getSNIContext(), getSNIContext(), g.getTranslationX(),
				g.getTranslationY(), matrix.getSNIContext(), GraphicsContext.OPAQUE, elapsedTime, colorMatrix));
	}

	private void checkReturnCode(int ret) {
		if (VectorGraphicsNatives.RET_SUCCESS != ret) {
			throw new VectorGraphicsException(ret);
		}
	}

	private long checkElapsedTime(long elapsedTime) {
		return XMath.limit(elapsedTime, 0, getDuration());
	}

	/* package */ void checkOverlapAlpha(int alpha) {
		if (this.hasOverlappingElements && (alpha != GraphicsContext.OPAQUE)) {
			throw new VectorGraphicsException(VectorGraphicsException.IMAGE_OVERLAPPING_ELEMENTS);
		}
	}

	/* package */ void checkOverlapAlpha(float[] colorMatrix) {
		if (20 > colorMatrix.length) {
			// array is too small, spec: throw ArrayIndexOutOfBoundsException
			throw new ArrayIndexOutOfBoundsException();
		}
		if (this.hasOverlappingElements && !ColorMatrixTransformer.keepsOpaque(colorMatrix)) {
			throw new VectorGraphicsException(VectorGraphicsException.IMAGE_OVERLAPPING_ELEMENTS);
		}
	}

	/* package */ byte[] getSNIContext() {
		return this.sniContext;
	}

}
