/*
 * Java
 *
 * Copyright 2021-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg;

import ej.microui.display.BufferedImage;

/**
 * Provides static shared native methods.
 */
public class VectorGraphicsNatives {

	/**
	 * Value returned on success.
	 */
	public static final int RET_SUCCESS = 0;

	private static boolean initialized = false;

	private VectorGraphicsNatives() {
		// Prevent instantiation.
	}

	/**
	 * Initializes the implementation. This is the first native called by the library.
	 */
	protected static void startup() {
		if (!initialized) {
			initialized = true;
			initialize();
		}
	}

	/**
	 * Initializes the implementation. This is the first native called by the library.
	 */
	private static native void initialize();

	/**
	 * Opens an image without allocating any data.
	 *
	 * @param path
	 *            the image path
	 * @param pathLength
	 *            the path length to prevent to load the full intern String array on the simulator
	 * @param sniContext
	 *            the context to use later to retrieve the loaded image
	 * @param metadata
	 *            some image metadata
	 * @return a {@link VectorGraphicsException} error code
	 */
	protected static native int getImage(byte[] path, int pathLength, byte[] sniContext, int[] metadata);

	/**
	 * Loads an image.
	 *
	 * If the image can be accessed directly, no memory will be allocated. Otherwise, memory will be allocated and the
	 * image will be copied into it. The caller is responsible for deallocating this memory area.
	 *
	 * @param path
	 *            the image path
	 * @param pathLength
	 *            the path length to prevent to load the full intern String array on the simulator
	 * @param sniContext
	 *            the context to use later to retrieve the loaded image
	 * @param metadata
	 *            some image metadata
	 * @return a {@link VectorGraphicsException} error code
	 */
	protected static native int loadImage(byte[] path, int pathLength, byte[] sniContext, int[] metadata);

	/**
	 * Creates an image from another image and applying a color deformation. This call allocates image data in the
	 * MicroUI heap.
	 *
	 * @param sniContextSource
	 *            the context of the image to duplicate
	 * @param sniContextDest
	 *            the context of the destination image to use later to retrieve the loaded image
	 * @param colorMatrix
	 *            the color matrix used to transform colors
	 * @return a {@link VectorGraphicsException} error code
	 */
	protected static native int createImage(byte[] sniContextSource, byte[] sniContextDest, float[] colorMatrix);

	/**
	 * Closes an image. This call deallocates image data in the MicroUI heap.
	 *
	 * @param sniContext
	 *            the context of the image to close
	 */
	public static native void closeImage(byte[] sniContext);

	/**
	 * Map the MicroVG BufferedVectorImage context on the MicroUI BufferedImage context.
	 *
	 * @param uiContext
	 *            the SNI context returned by {@link BufferedImage#getSNIContext()}
	 * @param vgContext
	 *            the new BufferedVectorImage context to map
	 */
	public static native void mapContext(byte[] uiContext, byte[] vgContext);

	/**
	 * Clears all commands stored in the buffered vector image.
	 *
	 * @param target
	 *            {@link BufferedImage#getSNIContext()}
	 */
	public static native void clear(byte[] target);
}
