/*
 * Java
 *
 * Copyright 2021-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg;

/**
 * Thrown to indicate that a method has caused an internal VectorGraphics error. This error is specified by an
 * identifier and this identifier can be retrieved calling {@link #getErrorCode()}.
 */
public class VectorGraphicsException extends RuntimeException {

	private static final long serialVersionUID = -7336803064067237535L;
	private static final String MICRO_VG = "MicroVG";

	/**
	 * Value returned on success.
	 */
	public static final int NO_ERROR = 0;

	/**
	 * Exception thrown when an image cannot be retrieved. Path must be relative to the application classpath and starts
	 * with '/'.
	 * <p>
	 * Value <code>-1</code> is assigned to <code>IMAGE_INVALID_PATH</code>.
	 */
	public static final int IMAGE_INVALID_PATH = -1;

	/**
	 * Exception thrown when an image cannot be drawn.
	 * <p>
	 * Value <code>-2</code> is assigned to <code>IMAGE_INVALID</code>.
	 */
	public static final int IMAGE_INVALID = -2;

	/**
	 * Exception thrown when an image with overlapping elements is drawn or filtered with a transparent alpha.
	 * <p>
	 * Value <code>-3</code> is assigned to <code>IMAGE_OVERLAPPING_ELEMENTS</code>.
	 */
	public static final int IMAGE_OVERLAPPING_ELEMENTS = -3;

	/**
	 * Exception thrown when a font cannot be retrieved. Path must be relative to the application classpath and starts
	 * with '/'.
	 * <p>
	 * Value <code>-4</code> is assigned to <code>FONT_INVALID_PATH</code>.
	 */
	public static final int FONT_INVALID_PATH = -4;

	/**
	 * Exception thrown when a font cannot be loaded.
	 * <p>
	 * Value <code>-5</code> is assigned to <code>FONT_INVALID</code>.
	 */
	public static final int FONT_INVALID = -5;

	/**
	 * Exception thrown when the application is trying to use a resource that has been closed.
	 * <p>
	 * Value <code>-6</code> is assigned to <code>RESOURCE_CLOSED</code>.
	 */
	public static final int RESOURCE_CLOSED = -6;

	/**
	 * Exception thrown when an image could not be loaded because of a parsing error.
	 * <p>
	 * Value <code>-7</code> is assigned to <code>IMAGE_PARSING_ERROR</code>.
	 */
	public static final int IMAGE_PARSING_ERROR = -7;

	/**
	 * Exception thrown when a font is loaded with complex text layout enabled but no complex layouter is available.
	 * <p>
	 * Value <code>-8</code> is assigned to <code>NO_COMPLEX_LAYOUTER_ERROR</code>.
	 */
	public static final int NO_COMPLEX_LAYOUTER_ERROR = -8;

	/**
	 * Exception thrown when an out of memory error occurs.
	 * <p>
	 * Value <code>-9</code> is assigned to <code>OUT_OF_MEMORY</code>.
	 */
	public static final int OUT_OF_MEMORY = -9;

	/**
	 * Exception thrown when the path initial length is too small.
	 */
	public static final int PATH_INVALID_LENGTH = 1;

	private final int errorCode;

	/**
	 * Constructs an exception with the specified error code.
	 *
	 * @param errorCode
	 *            the exception error code
	 */
	public VectorGraphicsException(int errorCode) {
		super(getMessage(errorCode));
		this.errorCode = errorCode;
	}

	/**
	 * Constructs an exception with the specified error code.
	 *
	 * @param errorCode
	 *            the exception error code
	 * @param cause
	 *            the cause
	 */
	public VectorGraphicsException(int errorCode, Exception cause) {
		super(getMessage(errorCode), cause);
		this.errorCode = errorCode;
	}

	private static String getMessage(int errorCode) {
		return MICRO_VG + ":E=" + errorCode;
	}

	/**
	 * Gets the reason why the exception has been thrown.
	 *
	 * @return the exception error code
	 */
	public int getErrorCode() {
		return this.errorCode;
	}

	/**
	 * Checks the value returned by the natives and throws the corresponding exception if required.
	 *
	 * @param ret
	 *            the native returned value
	 * @return the native returned value
	 * @throws VectorGraphicsException
	 *             if the native returned value is an error code.
	 */
	/* package */ static int checkNativeReturnedValue(int ret) {
		if (VectorGraphicsException.RESOURCE_CLOSED == ret) {
			throw new VectorGraphicsException(VectorGraphicsException.RESOURCE_CLOSED);
		}
		return ret;
	}

	/**
	 * Checks the value returned by the natives and throws the corresponding exception if required.
	 *
	 * @param ret
	 *            the native returned value
	 * @return the native returned value
	 * @throws VectorGraphicsException
	 *             if the native returned value is an error code.
	 */
	/* package */ static float checkNativeReturnedValue(float ret) {
		if (VectorGraphicsException.RESOURCE_CLOSED == ret) {
			throw new VectorGraphicsException(VectorGraphicsException.RESOURCE_CLOSED);
		}
		return ret;
	}

}
