/*
 * Java
 *
 * Copyright 2022-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg;

/**
 * Provides static native methods called by {@link VectorFont}.
 */
/* package */ class VectorFontNatives {

	static {
		VectorGraphicsNatives.startup();
	}

	private VectorFontNatives() {
		// Prevent instantiation.
	}

	/**
	 * Loads a font from a path.
	 *
	 * @param fontPath
	 *            the path of the font
	 * @param complexText
	 *            if true uses the complex text layouter.
	 * @return the face reference or 0 if the font could not be loaded.
	 */
	protected static native int loadFont(byte[] fontPath, boolean complexText);

	/**
	 * Measures the width of a text for the specified font and size.
	 *
	 * @param string
	 *            the character to measure
	 * @param face
	 *            the font face to use
	 * @param size
	 *            the font size, in pixels
	 * @param letterSpacing
	 *            the extra letter spacing to use
	 * @return the width of the specified character, in pixels or a negative error code on error.
	 */
	protected static native float stringWidth(char[] string, int face, float size, float letterSpacing);

	/**
	 * Measures the height of a text for the specified font and size.
	 *
	 * @param string
	 *            the character to measure
	 * @param face
	 *            the font face to use
	 * @param size
	 *            the font size, in pixels
	 * @return the width of the specified character, in pixels or a negative error code on error.
	 */
	protected static native float stringHeight(char[] string, int face, float size);

	/**
	 * Gets the baseline position for a font at the given font size.
	 *
	 * @param face
	 *            the font face to use
	 * @param size
	 *            the font size, in pixels
	 * @return the distance from the top of the font to the baseline, in pixels or a negative error code on error.
	 */
	protected static native float getBaselinePosition(int face, float size);

	/**
	 * Gets the height for a font at the given font size.
	 * <p>
	 * The height of a font is the distance between the max ascent line and the max descent line.
	 *
	 * @param face
	 *            the font face to use
	 * @param size
	 *            the font size, in pixels
	 * @return the distance from the top of the font to the bottom, in pixels or a negative error code on error.
	 */
	protected static native float getHeight(int face, float size);

	/**
	 * Disposes the resources associated to the font.
	 *
	 * @param face
	 *            the font face to use
	 */
	protected static native void dispose(int face);

	/**
	 * Determines if the platform as a complex layouter.
	 *
	 * @return if true a complex layouter is available
	 */
	protected static native boolean hasComplexLayouter();

}
