/*
 * Java
 *
 * Copyright 2020-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;

import ej.annotation.Nullable;
import ej.bon.ByteArray;
import ej.microui.MicroUIException;
import ej.microui.display.Font;
import ej.microui.display.Format;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;
import ej.sni.SNI;

/**
 * A Vector font.
 * <p>
 * <strong>Font metrics:</strong>
 * <p>
 * <ul>
 * <li><strong>Font size</strong>: the size of the text bounding box. Most of the glyphs will fit into that box, but
 * some may stick out of this box (like glyphs with accent, for example "Ä").</li>
 * <li><strong>Max ascent</strong>: the distance above the baseline for the highest glyph across the font.</li>
 * <li><strong>Baseline</strong>: the text baseline, a line on which the characters are placed.</li>
 * <li><strong>Max descent</strong>: the distance below the baseline for the lowest glyph across the font.</li>
 * </ul>
 * <p>
 * <strong>Important note:</strong> The font size should not be confused with the actual font height:
 * <ul>
 * <li>The font size specifies the height of the text bounding box, but some glyphs of the font may extend beyond the
 * box.</li>
 * <li>The actual font height is the distance between the line of maximum ascent and the line of maximum descent. All
 * the glyphs of the font will be fully enclosed between these two lines. The actual font height can be retrieved with
 * the method {@link #getHeight(float)}.</li>
 * </ul>
 */
public class VectorFont implements Closeable {

	/* package */ static final float DEFAULT_LETTER_SPACING = 0f;

	private static final int NOT_LOADED_FONT_INDEX = -1;
	private static final int UNLOADED = 0;
	private static final int INT_SIZE = 4;

	@Nullable
	private static VectorFont[] fonts;

	private final String path;
	private final boolean complexText;

	private int faceReference;

	private VectorFont(String fontPath, boolean complexText) {
		this.path = fontPath;
		this.complexText = complexText;
	}

	/**
	 * Loads a vector font from a path using a simple text layout. Equivalent to calling
	 * <code>loadFont(resourcePath, false)</code>.
	 *
	 * @param resourcePath
	 *            the path to get the font from
	 * @return a vector font
	 * @throws VectorGraphicsException
	 *             if the font could not be loaded (see {@link VectorGraphicsException#getErrorCode()} for the error
	 *             cause)
	 */
	public static VectorFont loadFont(String resourcePath) {
		return loadFont(resourcePath, false);
	}

	/**
	 * Loads a vector font from a path using the specified text layout(simple or complex).
	 *
	 * @param resourcePath
	 *            the path to get the font from
	 * @param complexText
	 *            if true the font layouter considers complex text layout features like contextual glyph substitution or
	 *            positioning.
	 *            <p>
	 *            Arabic and Thai scripts are examples of scripts that need complex text layout features
	 *            <p>
	 *            <ul>
	 *            <li>Simple text layout uses the glyph advance metrics and the font kerning table.</li>
	 *            <li>Complex text layout uses the font GPOS and GSUB tables.</li>
	 *            </ul>
	 *            <p>
	 *            The vector font file should contain the tables needed by the selected text layout.
	 *
	 * @return a vector font
	 * @throws VectorGraphicsException
	 *             if the font could not be loaded or if complexText is true and no complex layouter is available(see
	 *             {@link VectorGraphicsException#getErrorCode()} for the error cause)
	 *
	 */
	public static VectorFont loadFont(String resourcePath, boolean complexText) {
		return getFont(resourcePath, complexText);
	}

	/**
	 * Returns whether this font has been closed or not.
	 *
	 * @return {@code true} if the font has been closed, {@code false} otherwise
	 */
	public boolean isClosed() {
		return UNLOADED == getFaceReference();
	}

	/**
	 * Closes this font and its associated resources.
	 * <p>
	 * Calling this method on a font which has already been closed has no effect.
	 */
	@Override
	public void close() {

		int faceReference;
		synchronized (this) {
			faceReference = this.faceReference;
			// tag this font as unloaded immediately
			this.faceReference = UNLOADED;
		}

		if (UNLOADED != faceReference) {
			// remove this font from the array to be able to reload it
			VectorFont[] loadedFonts = getFonts();
			fonts = removeFont(loadedFonts, this);

			// close font's native resources
			VectorFontNatives.dispose(faceReference);
		}
	}

	/**
	 * Gets a vector font from a path.
	 *
	 * @param resourcePath
	 *            the path to get the font from.
	 * @param complexText
	 *            if true uses the complex text layouter.
	 *
	 * @return a vector font.
	 * @throws VectorGraphicsException
	 *             if the font could not be loaded (see {@link VectorGraphicsException#getErrorCode()} for the error
	 *             cause)
	 */
	private static synchronized VectorFont getFont(String resourcePath, boolean complexText) {
		VectorFont[] loadedFonts = getFonts();
		VectorFont font = getFontInternal(loadedFonts, resourcePath, complexText);
		if (font != null) {
			return font;
		}

		font = new VectorFont(resourcePath, complexText);
		font.load(complexText);
		VectorFont.fonts = addFont(loadedFonts, font);

		return font;
	}

	private static VectorFont[] getFonts() {
		// use lazy init rather than clinit since VectorFont.fonts is in context local storage
		if (VectorFont.fonts == null) {
			VectorFont.fonts = new VectorFont[0];
		}

		assert VectorFont.fonts != null;
		return VectorFont.fonts;
	}

	@Nullable
	private static VectorFont getFontInternal(VectorFont[] fonts, String path, boolean complexText) {
		VectorFont font = null;
		final int index = getFontIndex(fonts, path, complexText);
		if (index != NOT_LOADED_FONT_INDEX) {
			font = fonts[index];
		}

		return font;
	}

	private static int getFontIndex(VectorFont[] fonts, String path, boolean complexText) {
		int length = fonts.length;
		for (int i = 0; i < length; i++) {
			if (fonts[i].path.equals(path) && (complexText == fonts[i].complexText)) {
				return i;
			}
		}

		return NOT_LOADED_FONT_INDEX;
	}

	private static VectorFont[] addFont(VectorFont[] fonts, final VectorFont font) {
		int length = fonts.length;
		VectorFont[] result = new VectorFont[length + 1];
		System.arraycopy(fonts, 0, result, 0, length);
		result[length] = font;
		return result;
	}

	private static VectorFont[] removeFont(VectorFont[] fonts, final VectorFont font) {
		int index = getFontIndex(fonts, font.path, font.complexText);
		assert NOT_LOADED_FONT_INDEX != index;

		int length = fonts.length;
		VectorFont[] result = new VectorFont[length - 1];
		if (0 < index) {
			System.arraycopy(fonts, 0, result, 0, index);
		}
		length -= (index + 1);
		if (0 < length) {
			System.arraycopy(fonts, index + 1, result, index, length);
		}
		return result;
	}

	private void load(boolean complexText) {
		if (this.path.length() == 0 || this.path.charAt(0) != '/') {
			throw new VectorGraphicsException(VectorGraphicsException.FONT_INVALID_PATH);
		}

		byte[] pathCString = SNI.toCString(this.path);
		assert pathCString != null;

		this.faceReference = VectorFontNatives.loadFont(pathCString, complexText);

		if (UNLOADED == this.faceReference) {
			// an error has occurred: check if the font file exists or if it is a loading error

			try (InputStream resource = VectorFont.class.getResourceAsStream(this.path)) {
				if (resource == null) {
					// the resource cannot be found
					throw new VectorGraphicsException(VectorGraphicsException.FONT_INVALID_PATH);
				}
			} catch (IOException e) {
				// resource found but cannot read it
			}

			// check if load failed because complexText requested but no complexLayouter
			if (complexText && !(VectorFontNatives.hasComplexLayouter())) {
				throw new VectorGraphicsException(VectorGraphicsException.NO_COMPLEX_LAYOUTER_ERROR);
			}

			// resource found but cannot read it or cannot load it
			throw new VectorGraphicsException(VectorGraphicsException.FONT_INVALID);
		}
	}

	/**
	 * Returns the height of this font, at given font size.
	 * <p>
	 * The returned value is the distance between the line of maximum ascent and the line of maximum descent (see
	 * {@link VectorFont}).
	 * <p>
	 * This method can be used to size a text area. Since all the glyphs of a font are located between these two lines,
	 * any text will fit in an area of that height when drawn with {@link VectorGraphicsPainter} methods. If, instead,
	 * you prefer to adjust the size of your area to be the exact size of a specific text, refer to the method
	 * {@link #measureStringHeight(String, float)}.
	 * <p>
	 * The specified font size must be positive. If it is less than or equal to 0, the method will return 0.
	 *
	 * @param size
	 *            the font size, in pixels
	 * @return the height of this font at given font size, in pixels
	 * @throws VectorGraphicsException
	 *             if the measure could not be obtained for any reason (see
	 *             {@link VectorGraphicsException#getErrorCode()})
	 * @see #measureStringHeight(String, float)
	 * @see VectorGraphicsPainter#drawString(ej.microui.display.GraphicsContext, String, VectorFont, float, float,
	 *      float)
	 */
	public float getHeight(float size) {
		return VectorGraphicsException.checkNativeReturnedValue(VectorFontNatives.getHeight(this.faceReference, size));
	}

	/**
	 * Returns the position of the baseline for this font, at given font size.
	 * <p>
	 * The returned value is the distance between the line of maximum ascent and the baseline (see {@link VectorFont}).
	 * <p>
	 * The specified font size must be positive. If it is less than or equal to 0, the method will return 0.
	 *
	 * @param size
	 *            the font size, in pixels
	 * @return the baseline position of this font at given font size, in pixels
	 * @throws VectorGraphicsException
	 *             if the measure could not be obtained for any reason (see
	 *             {@link VectorGraphicsException#getErrorCode()})
	 */
	public float getBaselinePosition(float size) {
		return VectorGraphicsException
				.checkNativeReturnedValue(VectorFontNatives.getBaselinePosition(this.faceReference, size));
	}

	/**
	 * Returns the width of a string when it is drawn with this font and the given size.
	 * <p>
	 * The returned value is the width of the smallest rectangle that encloses all the glyphs.
	 * <p>
	 * This method can be used to size a text area. The given text will fit perfectly in an area of that width when
	 * drawn with {@link VectorGraphicsPainter} methods.
	 * <p>
	 * Note that the measure includes the gaps between the glyphs (also known as side bearings). For this reason, the
	 * measurement for a given string is not equal to the sum of the measurements for each glyph of that string.
	 * <p>
	 * The specified font size must be positive. If it is less than or equal to 0, the method will return 0.
	 * <p>
	 * Equivalent to calling {@link #measureStringWidth(String, float, float)} with a letter spacing of 0.
	 *
	 * @param string
	 *            the string to measure
	 * @param size
	 *            the font size, in pixels
	 * @return the width of the specified string, in pixels
	 * @throws VectorGraphicsException
	 *             if the measure could not be obtained for any reason (see
	 *             {@link VectorGraphicsException#getErrorCode()})
	 * @see #measureStringWidth(String, float, float)
	 * @see VectorGraphicsPainter#drawString(ej.microui.display.GraphicsContext, String, VectorFont, float, float,
	 *      float)
	 */
	public float measureStringWidth(String string, float size) {
		return measureStringWidth(string, size, DEFAULT_LETTER_SPACING);
	}

	/**
	 * Returns the width of a string when it is drawn with this font and the given size.
	 * <p>
	 * The returned value is the width of the smallest rectangle that encloses all the glyphs, taking into account the
	 * given extra letter spacing.
	 * <p>
	 * This method can be used to size a text area. The given text will fit perfectly in an area of that width when
	 * drawn with {@link VectorGraphicsPainter} methods.
	 * <p>
	 * Note that the measure includes the gaps between the glyphs (side bearings and extra letter spacing). For this
	 * reason, the measurement for a given string is not equal to the sum of the measurements for each glyph of that
	 * string.
	 * <p>
	 * The letter spacing argument specifies the extra space to add between the characters. A positive value will move
	 * the characters apart, a negative value will move them together. The default value is 0.
	 * <p>
	 * The specified font size must be positive. If it is less than or equal to 0, the method will return 0.
	 *
	 * @param string
	 *            the string to measure
	 * @param size
	 *            the font size, in pixels
	 * @param letterSpacing
	 *            the extra letter spacing to use, in pixels
	 * @return the width of the specified string, in pixels
	 * @throws VectorGraphicsException
	 *             if the measure could not be obtained for any reason (see
	 *             {@link VectorGraphicsException#getErrorCode()})
	 * @see #measureStringWidth(String, float)
	 * @see VectorGraphicsPainter#drawString(ej.microui.display.GraphicsContext, String, VectorFont, float, Matrix, int,
	 *      BlendMode, float)
	 */
	public float measureStringWidth(String string, float size, float letterSpacing) {
		return charsWidth(string.toCharArray(), size, letterSpacing);
	}

	/**
	 * Returns the height of a string when it is drawn with this font and the given size.
	 * <p>
	 * The returned value is the height of the smallest rectangle that encloses all the glyphs.
	 * <p>
	 * This method can be used to size a text area. The given text will fit perfectly in an area of that height when
	 * drawn with {@link VectorGraphicsPainter} methods. If, instead, you prefer to adjust the size of your area to fit
	 * any kind of text, refer to the method {@link #getHeight(float)}.
	 * <p>
	 * The specified font size must be positive. If it is less than or equal to 0, the method will return 0.
	 *
	 * @param string
	 *            the string to measure
	 * @param size
	 *            the font size, in pixels
	 * @return the height of the specified string, in pixels
	 * @throws VectorGraphicsException
	 *             if the measure could not be obtained for any reason (see
	 *             {@link VectorGraphicsException#getErrorCode()})
	 * @see #getHeight(float)
	 * @see VectorGraphicsPainter#drawString(ej.microui.display.GraphicsContext, String, VectorFont, float, float,
	 *      float)
	 */
	public float measureStringHeight(String string, float size) {
		return VectorGraphicsException.checkNativeReturnedValue(
				VectorFontNatives.stringHeight(string.toCharArray(), this.faceReference, size));
	}

	/**
	 * Determines if the font is loaded for complex text layout
	 *
	 * @return true is the font is configured for complex text layout otherwise false
	 */
	/* package */ boolean isComplexTextEnabled() {
		return this.complexText;
	}

	/**
	 * Gets the reference of the loaded font face.
	 *
	 * @return an integer representing the face address in C native code.
	 */
	/* package */ int getFaceReference() {
		return this.faceReference;
	}

	/**
	 * Retrieves a fixed-sized font from this vector font.
	 *
	 * @param size
	 *            the desired size
	 * @return the fixed-sized font
	 * @throws IllegalArgumentException
	 *             if the given size is lower of equals to 0
	 * @see Painter#drawString(ej.microui.display.GraphicsContext, String, Font, int, int)
	 */
	public Font getFont(int size) {
		if (isValidFontSize(size)) {
			return new VectorUiFont(size);
		} else {
			throw new IllegalArgumentException();
		}
	}

	private float charsWidth(char[] chars, float size, float letterSpacing) {
		return VectorGraphicsException.checkNativeReturnedValue(
				VectorFontNatives.stringWidth(chars, this.faceReference, size, letterSpacing));
	}

	// The translation of the graphics context is already applied to the given x and y.
	/* package */ void drawChars(GraphicsContext g, char[] chars, VectorFont font, // NOSONAR Avoid creating a type
			float size, float x, float y, @Nullable Matrix matrix, int alpha, BlendMode blendMode,
			float letterSpacing) {
		VectorGraphicsException
				.checkNativeReturnedValue(PainterNatives.drawString(g.getSNIContext(), chars, font.getFaceReference(),
						size, x, y, matrix != null ? matrix.values : null, alpha, blendMode.ordinal(), letterSpacing));
	}

	/* package */ static boolean isValidFontSize(float size) {
		return size > 0;
	}

	private byte[] getFontData(int size) {
		byte[] result = new byte[INT_SIZE * 2];
		ByteArray.writeInt(result, 0, this.faceReference);
		ByteArray.writeInt(result, INT_SIZE, size);
		return result;
	}

	class VectorUiFont extends Font {

		private final int size;

		/* package */ VectorUiFont(int size) {
			super(Format.CUSTOM_7, VectorFont.this.getFontData(size));
			this.size = size;
		}

		@Override
		public int getHeight() {
			return ceil(VectorFont.this.getHeight(this.size));
		}

		@Override
		public int getBaselinePosition() {
			return ceil(VectorFont.this.getBaselinePosition(this.size));
		}

		/**
		 * Assumes that the given value is in the Short range (less than 32767).
		 */
		private int ceil(float value) {
			return Short.MAX_VALUE - (int) (Short.MAX_VALUE - value);
		}

		@Override
		public byte[] getSNIContext() {
			if (VectorFont.this.isClosed()) {
				// font factory has been closed
				throw new MicroUIException(MicroUIException.RESOURCE_CLOSED);
			}
			return super.getSNIContext();
		}
	}
}
