/*
 * Java
 *
 * Copyright 2022-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg;

import java.io.Closeable;

/**
 * Represents a vector image which requires dynamic allocation in order to be created.
 * <p>
 * A resource vector image must be closed in order to free the dynamic allocation used to store the image data.
 */
public class ResourceVectorImage extends VectorImage implements Closeable {

	private static final int RAW_FLAG_FREE_MEMORY_ON_CLOSE = 0x02;

	private boolean closed;

	private final boolean isCloseable;

	/* package */ ResourceVectorImage(VectorImage source, byte[] sniContext, boolean isCloseable) {
		super(source, sniContext);
		this.isCloseable = isCloseable;
	}

	/* package */ ResourceVectorImage(byte[] sniContext, int[] metadata) {
		super(sniContext, metadata);
		int flags = metadata[RAW_OFFSET_U32_FLAGS];
		this.isCloseable = (flags & RAW_FLAG_FREE_MEMORY_ON_CLOSE) == RAW_FLAG_FREE_MEMORY_ON_CLOSE;
	}

	/* package */ ResourceVectorImage(float width, float height) {
		super(width, height);
		this.isCloseable = true;
	}

	/**
	 * Throws an exception if the image has been closed
	 */
	protected void checkClosed() {
		if (isClosed()) {
			throw new VectorGraphicsException(VectorGraphicsException.RESOURCE_CLOSED);
		}
	}

	/**
	 * Returns whether this vector image has been closed or not.
	 *
	 * @return <code>true</code> if the vector image has been closed, <code>false</code> otherwise.
	 */
	public boolean isClosed() {
		return this.closed;
	}

	/**
	 * Closes this vector image and its associated resources.
	 * <p>
	 * Calling this method on a vector image which has already been closed has no effect.
	 */
	@Override
	public void close() {
		boolean closed = this.closed;
		if (this.isCloseable && !closed) {
			byte[] sniContext = getSNIContext();
			this.closed = true;
			VectorGraphicsNatives.closeImage(sniContext);
		}
	}

	/**
	 * Loads a vector image from a path.
	 * <p>
	 * This method can load images which are in the internal image format defined by the MicroVG implementation. The
	 * caller is responsible for calling {@link ResourceVectorImage#close()} on the returned image when it is no longer
	 * used.
	 *
	 * @param resourcePath
	 *            the path to get the image from
	 * @return a vector image
	 * @throws VectorGraphicsException
	 *             if the image could not be retrieved for any reason (see
	 *             {@link VectorGraphicsException#getErrorCode()})
	 * @since 1.2
	 */
	public static ResourceVectorImage loadImage(String resourcePath) {
		byte[] sniPath = resourcePathToSniPath(resourcePath);
		byte[] sniContext = createSNIContext();
		int[] metadata = new int[RAW_METADATA_SIZE];

		int err = VectorGraphicsNatives.loadImage(sniPath, sniPath.length, sniContext, metadata);
		if (0 > err) {
			// the native returns the same error codes than the library
			throw new VectorGraphicsException(err);
		}

		return new ResourceVectorImage(sniContext, metadata);
	}

	@Override
	public byte[] getSNIContext() {
		checkClosed();
		return super.getSNIContext();
	}

}
