/*
 * Java
 *
 * Copyright 2022-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg;

/**
 * Provides static native methods called by {@link Path}.
 */
public class PathNatives {

	static {
		VectorGraphicsNatives.startup();
	}

	private PathNatives() {
		// Prevent instantiation.
	}

	/**
	 * Initializes the path array.
	 * <p>
	 * If the implementation requires a bigger array, it must return the number of required bytes.
	 *
	 * @param path
	 *            the array to initialize
	 * @param length
	 *            the array size
	 * @return {@link VectorGraphicsNatives#RET_SUCCESS} on success or a value that indicates the minimal required size
	 *         of the array.
	 */
	/* package */ static native int initializePath(byte[] path, int length);

	/**
	 * Adds a command with 1 point parameter in the array.
	 * <p>
	 * If the data (command and its parameters) cannot be added (out of memory), the implementation must return the
	 * number of bytes required to add the data. The caller has to increase the array and call again the native method.
	 *
	 * @param path
	 *            the array to update with the command and parameters.
	 * @param length
	 *            the array length.
	 * @param cmd
	 *            the command to add in the array.
	 * @param x
	 *            the command data 1.
	 * @param y
	 *            the command data 2.
	 * @return {@link VectorGraphicsNatives#RET_SUCCESS} on success or a value that indicates the number of bytes the
	 *         array must be enlarged.
	 */
	/* package */ static native int appendPathCommand(byte[] path, int length, int cmd, float x, float y);

	/**
	 * Adds a command with 2 points parameter in the array.
	 *
	 * @see #appendPathCommand(byte[], int, int, float, float)
	 */
	/* package */ static native int appendPathCommand(byte[] path, int length, int cmd, float x1, float y1, float x2,
			float y2);

	/**
	 * Adds a command with 3 points parameter in the array.
	 *
	 * @see #appendPathCommand(byte[], int, int, float, float)
	 */
	/* package */ static native int appendPathCommand(byte[] path, int length, int cmd, float x1, float y1, float x2,
			float y2, float x3, float y3);

	/**
	 * Open again the closed path (the last path command is CMD_CLOSED for sure).
	 * <p>
	 * The implementation resets the data offset at the end of the previous command.
	 *
	 * @param path
	 *            the array to update.
	 */
	/* package */ static native void reopenPath(byte[] path);

}
