/*
 * Java
 *
 * Copyright 2021-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 * This File is a derivative work. Subject to §4 of the applicable Apache License, MicroEJ provides the above different license terms and conditions for use.
 *
 * // Copyright (C) 2006 The Android Open Source Project
 * //
 * // Licensed under the Apache License, Version 2.0 (the "License");
 * // you may not use this file except in compliance with the License.
 * // You may obtain a copy of the License at
 * //
 * //      http://www.apache.org/licenses/LICENSE-2.0
 * //
 * // Unless required by applicable law or agreed to in writing, software
 * // distributed under the License is distributed on an "AS IS" BASIS,
 * // WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * // See the License for the specific language governing permissions and
 * // limitations under the License.
 */
package ej.microvg;

import ej.bon.Constants;

/**
 * Represents a geometric path consisting of straight line segments, quadratic curves, and cubic curves.
 */
public class Path {

	private static final int CMD_CLOSE = 0;
	private static final int CMD_MOVE = 1;
	private static final int CMD_LINE = 3;
	private static final int CMD_LINE_REL = 4;
	private static final int CMD_QUAD = 5;
	private static final int CMD_QUAD_REL = 6;
	private static final int CMD_CUBIC = 7;
	private static final int CMD_CUBIC_REL = 8;

	private static final String PATH_LENGTH_CONSTANT = "com.microej.microvg.path.length";
	private static final String CHUNK_LENGTH_CONSTANT = "com.microej.microvg.chunk.length";

	private byte[] data;
	private float currentX;
	private float currentY;
	private float minX;
	private float maxX;
	private float minY;
	private float maxY;
	private float offsetX;
	private float offsetY;

	private boolean initialMove;
	private boolean closed;

	/**
	 * Creates an empty path.
	 */
	public Path() {
		this.data = createRawData();
	}

	private static byte[] createRawData() {
		int size = Constants.getInt(PATH_LENGTH_CONSTANT);
		byte[] data = new byte[size];
		int result = PathNatives.initializePath(data, size);
		if (VectorGraphicsNatives.RET_SUCCESS != result) {
			throw new VectorGraphicsException(VectorGraphicsException.PATH_INVALID_LENGTH);
		}
		return data;
	}

	private void updateMinMax(float x, float y) {
		float newX = x + this.offsetX;
		float newY = y + this.offsetY;
		if (this.initialMove) {
			this.minX = Math.min(this.minX, newX);
			this.maxX = Math.max(this.maxX, newX);
			this.minY = Math.min(this.minY, newY);
			this.maxY = Math.max(this.maxY, newY);
		} else {
			this.minX = newX;
			this.maxX = newX;
			this.minY = newY;
			this.maxY = newY;
		}
	}

	private void updateRelative(float x, float y) {
		this.currentX += x;
		this.currentY += y;
		updateMinMax(this.currentX, this.currentY);
	}

	private void updateAbsolute(float x, float y) {
		this.currentX = x;
		this.currentY = y;
		updateMinMax(this.currentX, this.currentY);
	}

	/**
	 * Resets the path to an empty one.
	 */
	public void reset() {
		this.data = createRawData();
		this.initialMove = false;
		this.closed = false;
		this.currentX = 0;
		this.currentY = 0;
		this.offsetX = 0;
		this.offsetY = 0;
		this.minX = 0;
		this.minY = 0;
		this.maxX = 0;
		this.maxY = 0;
	}

	/**
	 * Sets the beginning of the next contour to the point (x,y).
	 *
	 * @param x
	 *            the x-coordinate of the start of a new contour
	 * @param y
	 *            the y-coordinate of the start of a new contour
	 */
	public void moveTo(float x, float y) {
		checkClosed();
		while (!checkAppendCommand(PathNatives.appendPathCommand(this.data, this.data.length, CMD_MOVE,
				x + this.offsetX, y + this.offsetY))) {
			// have to call again the native
		}
		updateAbsolute(x, y);
		this.initialMove = true;
	}

	/**
	 * Sets the beginning of the next contour relative to the last point on the previous contour. If there is no
	 * previous contour, this is treated the same as {@link #moveTo(float, float)}.
	 *
	 * @param dx
	 *            the amount to add to the x-coordinate of the end of the previous contour, to specify the start of a
	 *            new contour
	 * @param dy
	 *            the amount to add to the y-coordinate of the end of the previous contour, to specify the start of a
	 *            new contour
	 */
	public void moveToRelative(float dx, float dy) {
		checkClosed();
		updateRelative(dx, dy);
		while (!checkAppendCommand(
				PathNatives.appendPathCommand(this.data, this.data.length, CMD_MOVE, this.currentX, this.currentY))) {
			// have to call again the native
		}
		this.initialMove = true;
	}

	private void checkState() {
		if (!this.initialMove) {
			moveTo(0, 0);
		} else {
			checkClosed();
		}
	}

	/**
	 * Adds a line from the last point to the specified point (x,y). If no {@link #moveTo(float, float)} call has been
	 * made for this contour, the first point is automatically set to (0,0).
	 *
	 * @param x
	 *            the x-coordinate of the end of a line
	 * @param y
	 *            the y-coordinate of the end of a line
	 */
	public void lineTo(float x, float y) {
		checkState();
		while (!checkAppendCommand(PathNatives.appendPathCommand(this.data, this.data.length, CMD_LINE,
				x + this.offsetX, y + this.offsetY))) {
			// have to call again the native
		}
		updateAbsolute(x, y);
	}

	/**
	 * Same as {@link #lineTo(float, float)}, but the coordinates are considered relative to the last point on this
	 * contour. If there is no previous point, then a <code>moveTo(0,0)</code> is inserted automatically.
	 *
	 * @param dx
	 *            the amount to add to the x-coordinate of the previous point on this contour, to specify a line
	 * @param dy
	 *            the amount to add to the y-coordinate of the previous point on this contour, to specify a line
	 */
	public void lineToRelative(float dx, float dy) {
		checkState();
		while (!checkAppendCommand(PathNatives.appendPathCommand(this.data, this.data.length, CMD_LINE_REL, dx, dy))) {
			// have to call again the native
		}
		updateRelative(dx, dy);
	}

	/**
	 * Adds a quadratic bezier from the last point, approaching control point (x1,y1), and ending at (x2,y2). If no
	 * {@link #moveTo(float, float)} call has been made for this contour, the first point is automatically set to (0,0).
	 *
	 * @param x1
	 *            the x-coordinate of the control point on a quadratic curve
	 * @param y1
	 *            the y-coordinate of the control point on a quadratic curve
	 * @param x2
	 *            the x-coordinate of the end point on a quadratic curve
	 * @param y2
	 *            the y-coordinate of the end point on a quadratic curve
	 */
	public void quadTo(float x1, float y1, float x2, float y2) {
		checkState();
		while (!checkAppendCommand(PathNatives.appendPathCommand(this.data, this.data.length, CMD_QUAD,
				x1 + this.offsetX, y1 + this.offsetY, x2 + this.offsetX, y2 + this.offsetY))) {
			// have to call again the native
		}
		updateMinMax(x1, y1);
		updateAbsolute(x2, y2);
	}

	/**
	 * Same as {@link #quadTo(float, float, float, float)}, but the coordinates are considered relative to the last
	 * point on this contour. If there is no previous point, then a moveTo(0,0) is inserted automatically.
	 *
	 * @param dx1
	 *            the amount to add to the x-coordinate of the last point on this contour, for the control point of a
	 *            quadratic curve
	 * @param dy1
	 *            the amount to add to the y-coordinate of the last point on this contour, for the control point of a
	 *            quadratic curve
	 * @param dx2
	 *            the amount to add to the x-coordinate of the last point on this contour, for the end point of a
	 *            quadratic curve
	 * @param dy2
	 *            the amount to add to the y-coordinate of the last point on this contour, for the end point of a
	 *            quadratic curve
	 */
	public void quadToRelative(float dx1, float dy1, float dx2, float dy2) {
		checkState();
		while (!checkAppendCommand(
				PathNatives.appendPathCommand(this.data, this.data.length, CMD_QUAD_REL, dx1, dy1, dx2, dy2))) {
			// have to call again the native
		}
		updateMinMax(this.currentX + dx1, this.currentY + dy1);
		updateRelative(dx2, dy2);
	}

	/**
	 * Adds a cubic bezier from the last point, approaching control points (x1,y1) and (x2,y2), and ending at (x3,y3).
	 * If no {@link #moveTo(float, float)} call has been made for this contour, the first point is automatically set to
	 * (0,0).
	 *
	 * @param x1
	 *            the x-coordinate of the 1st control point on a cubic curve
	 * @param y1
	 *            the y-coordinate of the 1st control point on a cubic curve
	 * @param x2
	 *            the x-coordinate of the 2nd control point on a cubic curve
	 * @param y2
	 *            the y-coordinate of the 2nd control point on a cubic curve
	 * @param x3
	 *            the x-coordinate of the end point on a cubic curve
	 * @param y3
	 *            the y-coordinate of the end point on a cubic curve
	 */
	public void cubicTo(float x1, float y1, float x2, float y2, float x3, float y3) {
		checkState();
		while (!checkAppendCommand(PathNatives.appendPathCommand(this.data, this.data.length, CMD_CUBIC,
				x1 + this.offsetX, y1 + this.offsetY, x2 + this.offsetX, y2 + this.offsetY, x3 + this.offsetX,
				y3 + this.offsetY))) {
			// have to call again the native
		}
		updateMinMax(x1, y1);
		updateMinMax(x2, y2);
		updateAbsolute(x3, y3);
	}

	/**
	 * Same as {@link #cubicTo(float, float, float, float, float, float)}, but the coordinates are considered relative
	 * to the current point on this contour. If there is no previous point, then a moveTo(0,0) is inserted
	 * automatically.
	 *
	 * @param dx1
	 *            the x-coordinate of the 1st control point on a cubic curve
	 * @param dy1
	 *            the y-coordinate of the 1st control point on a cubic curve
	 * @param dx2
	 *            the x-coordinate of the 2nd control point on a cubic curve
	 * @param dy2
	 *            the y-coordinate of the 2nd control point on a cubic curve
	 * @param dx3
	 *            the x-coordinate of the end point on a cubic curve
	 * @param dy3
	 *            the y-coordinate of the end point on a cubic curve
	 */
	public void cubicToRelative(float dx1, float dy1, float dx2, float dy2, float dx3, float dy3) {
		checkState();
		while (!checkAppendCommand(PathNatives.appendPathCommand(this.data, this.data.length, CMD_CUBIC_REL, dx1, dy1,
				dx2, dy2, dx3, dy3))) {
			// have to call again the native
		}
		updateMinMax(this.currentX + dx1, this.currentY + dy1);
		updateMinMax(this.currentX + dx2, this.currentY + dy2);
		updateRelative(dx3, dy3);
	}

	/**
	 * Closes the current contour. If the current point is not equal to the first point of the contour, a line segment
	 * is automatically added.
	 */
	public void close() {
		if (!this.closed) {
			while (!checkAppendCommand(PathNatives.appendPathCommand(this.data, this.data.length, CMD_CLOSE, this.minX,
					this.minY, this.maxX, this.maxY))) {
				// have to call again the native
			}
			this.closed = true;
		}
	}

	private void checkClosed() {
		if (this.closed) {
			// reopen by removing the command "close"
			PathNatives.reopenPath(this.data);
			this.closed = false;
		}
	}

	/**
	 * Sets the offset that will be applied to all the subsequent absolute points of the contour.
	 *
	 * @param dx
	 *            the amount in the x direction to offset the subsequent points
	 * @param dy
	 *            the amount in the y direction to offset the subsequent points
	 */
	public void setOrigin(float dx, float dy) {
		this.offsetX = dx;
		this.offsetY = dy;
	}

	/**
	 * Gets the left bound of this path.
	 * <p>
	 * If this path does not contain any point, all bounds are equal to 0.
	 *
	 * @return the left bound of this path
	 */
	public float getLeftBound() {
		return this.minX;
	}

	/**
	 * Gets the right bound of this path.
	 * <p>
	 * If this path does not contain any point, all bounds are equal to 0.
	 *
	 * @return the right bound of this path
	 */
	public float getRightBound() {
		return this.maxX;
	}

	/**
	 * Gets the top bound of this path.
	 * <p>
	 * If this path does not contain any point, all bounds are equal to 0.
	 *
	 * @return the top bound of this path
	 */
	public float getTopBound() {
		return this.minY;
	}

	/**
	 * Gets the bottom bound of this path.
	 * <p>
	 * If this path does not contain any point, all bounds are equal to 0.
	 *
	 * @return the bottom bound of this path
	 */
	public float getBottomBound() {
		return this.maxY;
	}

	/* package */ byte[] getData() {
		// ensure path is closed
		close();
		return this.data; // NOSONAR the data is used to be pushed to a native.
	}

	private boolean checkAppendCommand(int ret) {

		if (ret > 0) {

			// the buffer is too small: must enlarge it
			byte[] previousPath = this.data;
			int length = previousPath.length;
			byte[] newPath = new byte[length + Math.max(ret, Constants.getInt(CHUNK_LENGTH_CONSTANT))];
			System.arraycopy(previousPath, 0, newPath, 0, length);
			this.data = newPath;

			// caller must call again the native
			return false;
		}

		// buffer is large enough, the new command has been added
		return true;
	}

	/* package */ boolean isEmpty() {
		return !this.initialMove;
	}
}
