/*
 * Java
 *
 * Copyright 2021-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg;

import ej.annotation.Nullable;

/**
 * Provides static native methods called by {@link VectorGraphicsPainter}.
 */
public class PainterNatives {

	static {
		VectorGraphicsNatives.startup();
	}

	private PainterNatives() {
		// Prevent instantiation.
	}

	/**
	 * Draws a path filled with the color of the graphics context.
	 *
	 * @param gc
	 *            the destination
	 * @param path
	 *            the path's data
	 * @param x
	 *            the destination X coordinate
	 * @param y
	 *            the destination Y coordinate
	 * @param matrix
	 *            the deformation to apply
	 * @param fillRule
	 *            the fill type
	 * @param blend
	 *            the blend mode
	 * @param color
	 *            the color to apply (with an opacity)
	 * @return {@link VectorGraphicsNatives#RET_SUCCESS} on success
	 */
	public static native int drawPath(byte[] gc, byte[] path, int x, int y, float[] matrix, int fillRule, int blend,
			int color);

	/**
	 * Draws a path filled with a linear gradient.
	 *
	 * @param gc
	 *            the destination
	 * @param path
	 *            the path's data
	 * @param x
	 *            the destination X coordinate
	 * @param y
	 *            the destination Y coordinate
	 * @param matrix
	 *            the deformation to apply
	 * @param fillRule
	 *            the fill type
	 * @param alpha
	 *            the global opacity to apply
	 * @param blend
	 *            the blend mode
	 * @param gradient
	 *            the gradient to apply
	 * @param gradientMatrix
	 *            the gradient deformation, may be null (means identity)
	 * @return {@link VectorGraphicsNatives#RET_SUCCESS} on success
	 */
	public static native int drawGradient(byte[] gc, byte[] path, int x, int y, float[] matrix, int fillRule, int alpha,
			int blend, int[] gradient, @Nullable float[] gradientMatrix);

	/**
	 * Draws a string with the color of the graphics context.
	 *
	 * @param gc
	 *            the graphics context to draw on
	 * @param string
	 *            the array of characters to draw
	 * @param face
	 *            the font face to use
	 * @param size
	 *            the font size, in pixels
	 * @param x
	 *            the destination X coordinate
	 * @param y
	 *            the destination Y coordinate
	 * @param matrix
	 *            the transformation matrix to apply
	 * @param alpha
	 *            the global opacity rendering value
	 * @param blendMode
	 *            the blend mode to use
	 * @param letterSpacing
	 *            the extra letter spacing to use
	 * @return {@link VectorGraphicsNatives#RET_SUCCESS} on success
	 */
	public static native int drawString(byte[] gc, char[] string, int face, float size, float x, float y,
			@Nullable float[] matrix, int alpha, int blendMode, float letterSpacing);

	/**
	 * Draws a string with a linear gradient.
	 *
	 * @param gc
	 *            the graphics context to draw on
	 * @param string
	 *            the array of characters to draw
	 * @param face
	 *            the font face to use
	 * @param size
	 *            the font size, in pixels
	 * @param x
	 *            the destination X coordinate
	 * @param y
	 *            the destination Y coordinate
	 * @param matrix
	 *            the transformation matrix to apply
	 * @param alpha
	 *            the global opacity rendering value
	 * @param blendMode
	 *            the blend mode to use
	 * @param letterSpacing
	 *            the extra letter spacing to use
	 * @param gradient
	 *            the gradient to apply
	 * @param gradientMatrix
	 *            the gradient deformation, may be null (means identity)
	 * @return {@link VectorGraphicsNatives#RET_SUCCESS} on success
	 */
	public static native int drawGradientString(byte[] gc, char[] string, int face, float size, float x, float y,
			float[] matrix, int alpha, int blendMode, float letterSpacing, int[] gradient,
			@Nullable float[] gradientMatrix);

	/**
	 * Draws a string along a circle, with the color of the graphics context.
	 *
	 * @param gc
	 *            the graphics context to draw on
	 * @param string
	 *            the array of characters to draw
	 * @param face
	 *            the font face to use
	 * @param size
	 *            the font size, in pixels
	 * @param x
	 *            the destination X coordinate
	 * @param y
	 *            the destination Y coordinate
	 * @param matrix
	 *            the transformation matrix to apply
	 * @param alpha
	 *            the global opacity rendering value
	 * @param blendMode
	 *            the blend mode to use
	 * @param letterSpacing
	 *            the extra letter spacing to use
	 * @param radius
	 *            the radius of the circle
	 * @param direction
	 *            the direction of the text along the circle
	 * @return {@link VectorGraphicsNatives#RET_SUCCESS} on success
	 */
	public static native int drawStringOnCircle(byte[] gc, char[] string, int face, float size, int x, int y,
			float[] matrix, int alpha, int blendMode, float letterSpacing, float radius, int direction);

	/**
	 * Draws a string along a circle, with a linear gradient.
	 *
	 * @param gc
	 *            the graphics context to draw on
	 * @param string
	 *            the array of characters to draw
	 * @param face
	 *            the font face to use
	 * @param size
	 *            the font size, in pixels
	 * @param x
	 *            the destination X coordinate
	 * @param y
	 *            the destination Y coordinate
	 * @param matrix
	 *            the transformation matrix to apply
	 * @param alpha
	 *            the global opacity rendering value
	 * @param blendMode
	 *            the blend mode to use
	 * @param letterSpacing
	 *            the extra letter spacing to use
	 * @param radius
	 *            the radius of the circle
	 * @param direction
	 *            the direction of the text along the circle
	 * @param gradient
	 *            the gradient to apply
	 * @param gradientMatrix
	 *            the gradient deformation, may be null (means identity)
	 * @return {@link VectorGraphicsNatives#RET_SUCCESS} on success
	 */
	public static native int drawGradientStringOnCircle(byte[] gc, char[] string, int face, float size, int x, int y,
			float[] matrix, int alpha, int blendMode, float letterSpacing, float radius, int direction, int[] gradient,
			@Nullable float[] gradientMatrix);

	/**
	 * Draws an image with transformation and opacity. Optionally apply an animation and / or a color filtering.
	 *
	 * @param gc
	 *            the graphics context to draw on
	 * @param image
	 *            the image to draw
	 * @param x
	 *            the destination X coordinate
	 * @param y
	 *            the destination Y coordinate
	 * @param matrix
	 *            the transformation matrix to apply
	 * @param alpha
	 *            the global opacity rendering value
	 * @param elapsedTime
	 *            the elapsed time since the beginning of the animation, in milliseconds
	 * @param colorMatrix
	 *            the color matrix used to transform colors
	 * @return a {@link VectorGraphicsException} error code
	 */
	public static native int drawImage(byte[] gc, byte[] image, int x, int y, float[] matrix, int alpha,
			long elapsedTime, @Nullable float[] colorMatrix);

}
