/*
 * Java
 *
 * Copyright 2021-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg;

/**
 * Provides static native methods called by {@link Matrix}
 */
/* package */ class MatrixNatives {

	static {
		VectorGraphicsNatives.startup();
	}

	private MatrixNatives() {
		// Prevent instantiation.
	}

	/**
	 * Creates an identity matrix.
	 *
	 * @param matrix
	 *            the matrix to initialize
	 */
	/* package */ static native void identity(float[] matrix);

	/**
	 * Deep copies the given matrix into destination.
	 *
	 * @param dest
	 *            the matrix to copy to
	 * @param src
	 *            the matrix to copy from
	 */
	/* package */ static native void copy(float[] dest, float[] src);

	/**
	 * Sets the matrix to translate by (dx, dy).
	 *
	 * @param matrix
	 *            the matrix to translate
	 * @param dx
	 *            the translation on x
	 * @param dy
	 *            the translation on y
	 */
	/* package */ static native void setTranslate(float[] matrix, float dx, float dy);

	/**
	 * Sets the matrix to scale by (sx, sy).
	 *
	 * @param matrix
	 *            the matrix to scale
	 *
	 * @param sx
	 *            the scale on x
	 * @param sy
	 *            the scale on y
	 */
	/* package */ static native void setScale(float[] matrix, float sx, float sy);

	/**
	 * Sets the matrix to rotate about (0,0) by the specified number of degrees.
	 *
	 * @param matrix
	 *            the matrix to rotate
	 * @param degrees
	 *            the degrees to rotate
	 */
	/* package */ static native void setRotate(float[] matrix, float degrees);

	/**
	 * Sets the matrix to the concatenation of the two specified matrices.
	 * <p>
	 * Either of the two matrices may also be the target matrix, that is
	 * <code>matrixA.setConcat(matrixA, matrixB);</code> is valid.
	 *
	 * @param dest
	 *            the destination
	 * @param a
	 *            the first matrix
	 * @param b
	 *            the second matrix
	 */
	/* package */ static native void setConcat(float[] dest, float[] a, float[] b);

	/**
	 * Preconcats the matrix with the specified translation. M' = M * T(dx, dy)
	 *
	 * @param matrix
	 *            the matrix to translate
	 * @param x
	 *            the translation on x
	 * @param y
	 *            the translation on y
	 */
	/* package */ static native void translate(float[] matrix, float x, float y);

	/**
	 * Preconcats the matrix with the specified scale. M' = M * S(sx, sy)
	 *
	 * @param matrix
	 *            the matrix to scale
	 * @param scaleX
	 *            the scale on x
	 * @param scaleY
	 *            the scale on y
	 */
	/* package */ static native void scale(float[] matrix, float scaleX, float scaleY);

	/**
	 * Preconcats the matrix with the specified rotation. M' = M * R(degrees)
	 *
	 * @param matrix
	 *            the matrix to rotate
	 * @param degrees
	 *            the degrees to rotate
	 */
	/* package */ static native void rotate(float[] matrix, float degrees);

	/**
	 * Preconcats the matrix with the specified matrix. M' = M * other
	 *
	 * @param matrix
	 *            the matrix to concat
	 * @param other
	 *            the matrix to concat with
	 */
	/* package */ static native void concatenate(float[] matrix, float[] other);

	/**
	 * Postconcats the matrix with the specified translation. M' = T(dx, dy) * M
	 *
	 * @param matrix
	 *            the matrix to translate
	 * @param dx
	 *            the translation on x
	 * @param dy
	 *            the translation on y
	 */
	/* package */ static native void postTranslate(float[] matrix, float dx, float dy);

	/**
	 * Postconcats the matrix with the specified scale. M' = S(sx, sy) * M
	 *
	 * @param matrix
	 *            the matrix to scale
	 * @param sx
	 *            the scale on x
	 * @param sy
	 *            the scale on y
	 */
	/* package */ static native void postScale(float[] matrix, float sx, float sy);

	/**
	 * Postconcats the matrix with the specified rotation. M' = R(degrees) * M
	 *
	 * @param matrix
	 *            the matrix to rotate
	 * @param degrees
	 *            the degrees to rotate
	 */
	/* package */ static native void postRotate(float[] matrix, float degrees);

	/**
	 * Postconcats the matrix with the specified matrix. M' = other * M
	 *
	 * @param matrix
	 *            the matrix to concat
	 * @param other
	 *            the matrix to concat with
	 */
	/* package */ static native void postConcat(float[] matrix, float[] other);
}
