/*
 * Java
 *
 * Copyright 2021-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 * This File is a derivative work. Subject to §4 of the applicable Apache License, MicroEJ provides the above different license terms and conditions for use.
 *
 * // Copyright (C) 2006 The Android Open Source Project
 * //
 * // Licensed under the Apache License, Version 2.0 (the "License");
 * // you may not use this file except in compliance with the License.
 * // You may obtain a copy of the License at
 * //
 * //      http://www.apache.org/licenses/LICENSE-2.0
 * //
 * // Unless required by applicable law or agreed to in writing, software
 * // distributed under the License is distributed on an "AS IS" BASIS,
 * // WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * // See the License for the specific language governing permissions and
 * // limitations under the License.
 */
package ej.microvg;

/**
 * Represents a 3x3 matrix for transforming coordinates.
 */
public class Matrix {

	private static final int MATRIX_SIZE = 3 * 3;

	/** The flat transformation matrix [m00 m01 m02 m10 m11 m12 m20 m21 m22]. */
	public final float[] values;

	/**
	 * Creates an identity matrix.
	 */
	public Matrix() {
		this.values = new float[MATRIX_SIZE];
		MatrixNatives.identity(this.values);
	}

	/**
	 * Creates a matrix that is a deep copy of the given matrix.
	 *
	 * @param source
	 *            the matrix to copy from
	 */
	public Matrix(Matrix source) {
		this();
		MatrixNatives.copy(this.values, source.values);
	}

	/**
	 * Deep copies the given matrix into this matrix.
	 *
	 * @param source
	 *            the matrix to copy from
	 */
	public void set(Matrix source) {
		MatrixNatives.copy(this.values, source.values);
	}

	/**
	 * Resets the matrix to identity.
	 */
	public void reset() {
		MatrixNatives.identity(this.values);
	}

	/**
	 * Sets the matrix to translate by (dx, dy).
	 *
	 * @param dx
	 *            the translation on x
	 * @param dy
	 *            the translation on y
	 */
	public void setTranslate(float dx, float dy) {
		MatrixNatives.setTranslate(this.values, dx, dy);
	}

	/**
	 * Sets the matrix to scale by (sx, sy).
	 *
	 * @param sx
	 *            the scale on x
	 * @param sy
	 *            the scale on y
	 */
	public void setScale(float sx, float sy) {
		MatrixNatives.setScale(this.values, sx, sy);
	}

	/**
	 * Sets the matrix to rotate about (0,0) by the specified number of degrees.
	 *
	 * @param degrees
	 *            the degrees to rotate
	 */
	public void setRotate(float degrees) {
		MatrixNatives.setRotate(this.values, degrees);
	}

	/**
	 * Sets the matrix to the concatenation of the two specified matrices.
	 * <p>
	 * Either of the two matrices may also be the target matrix, that is
	 * <code>matrixA.setConcat(matrixA, matrixB);</code> is valid.
	 *
	 * @param a
	 *            the first matrix
	 * @param b
	 *            the second matrix
	 */
	public void setConcat(Matrix a, Matrix b) {
		MatrixNatives.setConcat(this.values, a.values, b.values);
	}

	/**
	 * Preconcats the matrix with the specified translation. M' = M * T(dx, dy)
	 *
	 * @param dx
	 *            the translation on x
	 * @param dy
	 *            the translation on y
	 */
	public void preTranslate(float dx, float dy) {
		MatrixNatives.translate(this.values, dx, dy);
	}

	/**
	 * Preconcats the matrix with the specified scale. M' = M * S(sx, sy)
	 *
	 * @param sx
	 *            the scale on x
	 * @param sy
	 *            the scale on y
	 */
	public void preScale(float sx, float sy) {
		MatrixNatives.scale(this.values, sx, sy);
	}

	/**
	 * Preconcats the matrix with the specified rotation. M' = M * R(degrees)
	 *
	 * @param degrees
	 *            the degrees to rotate
	 */
	public void preRotate(float degrees) {
		MatrixNatives.rotate(this.values, degrees);
	}

	/**
	 * Preconcats the matrix with the specified matrix. M' = M * other
	 *
	 * @param other
	 *            the matrix to concat with
	 */
	public void preConcat(Matrix other) {
		MatrixNatives.concatenate(this.values, other.values);
	}

	/**
	 * Postconcats the matrix with the specified translation. M' = T(dx, dy) * M
	 *
	 * @param dx
	 *            the translation on x
	 * @param dy
	 *            the translation on y
	 */
	public void postTranslate(float dx, float dy) {
		MatrixNatives.postTranslate(this.values, dx, dy);
	}

	/**
	 * Postconcats the matrix with the specified scale. M' = S(sx, sy) * M
	 *
	 * @param sx
	 *            the scale on x
	 * @param sy
	 *            the scale on y
	 */
	public void postScale(float sx, float sy) {
		MatrixNatives.postScale(this.values, sx, sy);
	}

	/**
	 * Postconcats the matrix with the specified rotation. M' = R(degrees) * M
	 *
	 * @param degrees
	 *            the degrees to rotate
	 */
	public void postRotate(float degrees) {
		MatrixNatives.postRotate(this.values, degrees);
	}

	/**
	 * Postconcats the matrix with the specified matrix. M' = other * M
	 *
	 * @param other
	 *            the matrix to concat with
	 */
	public void postConcat(Matrix other) {
		MatrixNatives.postConcat(this.values, other.values);
	}

	/**
	 * Returns the SNI context data of this matrix.
	 * <p>
	 * The SNI context can be used to call a native method with SNI. This allows to identify and to use an matrix in the
	 * native world in order to perform operations with this matrix.
	 * <p>
	 * The data format is implementation specific.
	 *
	 * @return the SNI context of this matrix.
	 */
	public float[] getSNIContext() {
		return this.values;
	}

}
