/*
 * Java
 *
 * Copyright 2021-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 * This File is a derivative work. Subject to §4 of the applicable Apache License, MicroEJ provides the above different license terms and conditions for use.
 *
 * // Copyright (C) 2006 The Android Open Source Project
 * //
 * // Licensed under the Apache License, Version 2.0 (the "License");
 * // you may not use this file except in compliance with the License.
 * // You may obtain a copy of the License at
 * //
 * //      http://www.apache.org/licenses/LICENSE-2.0
 * //
 * // Unless required by applicable law or agreed to in writing, software
 * // distributed under the License is distributed on an "AS IS" BASIS,
 * // WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * // See the License for the specific language governing permissions and
 * // limitations under the License.
 */
package ej.microvg;

import ej.annotation.Nullable;

/**
 * Represents a linear gradient on a vector.
 */
public class LinearGradient {

	/* package */ final int[] sniContext;
	/* package */ final Matrix matrix;

	/**
	 * Creates a linear gradient.
	 *
	 * @param xStart
	 *            the x-coordinate for the start of the gradient vector
	 * @param yStart
	 *            the y-coordinate for the start of the gradient vector
	 * @param xEnd
	 *            the x-coordinate for the end of the gradient vector
	 * @param yEnd
	 *            the y-coordinate for the end of the gradient vector
	 * @param colors
	 *            the sRGB colors to be distributed along the gradient vector
	 * @param positions
	 *            the relative positions [0..1] of each corresponding color in the colors array
	 */
	public LinearGradient(float xStart, float yStart, float xEnd, float yEnd, int[] colors, float[] positions) {
		this.matrix = new Matrix();
		this.sniContext = createRawData(xStart, yStart, xEnd, yEnd, colors, positions);
	}

	/**
	 * Creates a linear gradient.
	 *
	 * @param colors
	 *            the sRGB colors to be distributed along the gradient vector
	 * @param xStart
	 *            the x-coordinate for the start of the gradient vector
	 * @param yStart
	 *            the y-coordinate for the start of the gradient vector
	 * @param xEnd
	 *            the x-coordinate for the end of the gradient vector
	 * @param yEnd
	 *            the y-coordinate for the end of the gradient vector
	 */
	public LinearGradient(float xStart, float yStart, float xEnd, float yEnd, int[] colors) {
		this.matrix = new Matrix();
		this.sniContext = createRawData(xStart, yStart, xEnd, yEnd, colors, null);
	}

	/* package */ static int[] createRawData(float xStart, float yStart, float xEnd, float yEnd, int[] colors,
			@Nullable float[] positions) {

		int count = colors.length;

		if ((xStart == xEnd) && (yStart == yEnd)) {
			// the gradient length is null, create a horizontal gradient using first and last color.
			// the transition point is placed at the center of the scale.
			colors[1] = colors[count - 1];
			count = 2;
			positions = (null == positions) ? new float[count] : positions;
			positions[0] = 0;
			positions[1] = 1;
			xStart -= 0.5f;
			xEnd += 0.5f;
		} else if (null == positions) {
			positions = new float[count];
			float step = 1.0f / (count - 1);
			for (int i = 0; i < count; i++) {
				positions[i] = i * step;
			}
		}

		int size = count * 2 /* at least colors and positions */ ;
		int[] data;
		do {
			data = new int[size];
			size = GradientNatives.initializeGradient(data, data.length, colors, count, positions, xStart, yStart, xEnd,
					yEnd);
		} while (VectorGraphicsNatives.RET_SUCCESS != size);
		return data;
	}

	/**
	 * Gets the matrix of this linear gradient.
	 * <p>
	 * Any modification on the matrix will affect the gradient.
	 *
	 * @return the matrix
	 */
	public Matrix getMatrix() {
		return this.matrix;
	}
}
