/*
 * Java
 *
 * Copyright 2022-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg;

import ej.microui.display.GraphicsContext;
import ej.microvg.image.ColorMatrixTransformer;
import ej.microvg.image.ImageClip;
import ej.microvg.image.ImageElement;

/**
 * Vector image with Java objects {@link ImageElement}. This class is temporarily kept for convenience. It overrides all
 * drawings and derivation methods.
 */
public class InternalVectorImage extends ResourceVectorImage {

	private final ImageElement[] paths;

	private boolean closed;

	/* package */ InternalVectorImage(float width, float height, ImageElement[] paths) {
		super(width, height);
		this.paths = paths;

	}

	private InternalVectorImage(InternalVectorImage source, ColorMatrixTransformer colorMatrixTransformer) {
		super(source.getWidth(), source.getHeight());
		int length = source.paths.length;
		this.paths = new ImageElement[length];
		for (int i = 0; i < length; i++) {
			this.paths[i] = source.paths[i].transform(colorMatrixTransformer);
		}
	}

	@Override
	public ResourceVectorImage filterImage(float[] colorMatrix) {
		checkClosed();
		checkOverlapAlpha(colorMatrix);
		return new InternalVectorImage(this, new ColorMatrixTransformer(colorMatrix));
	}

	/**
	 * Draws the paths with given matrix.
	 *
	 * @param g
	 *            the graphics context to draw on.
	 * @param matrix
	 *            the matrix to apply
	 */
	@Override
	/* package */ void draw(GraphicsContext g, Matrix matrix) {
		checkClosed();
		ImageClip imageClip = preDraw(g, matrix);

		ImageElement[] paths = this.paths;
		for (ImageElement path : paths) {
			path.draw(g, matrix);
		}

		postDraw(g, imageClip);
	}

	/**
	 * Draws the paths with given matrix.
	 *
	 * @param g
	 *            the graphics context to draw on
	 * @param matrix
	 *            the matrix to apply
	 * @param alpha
	 *            the global opacity rendering value
	 */
	@Override
	/* package */ void draw(GraphicsContext g, Matrix matrix, int alpha) {
		checkClosed();
		checkOverlapAlpha(alpha);
		ImageClip imageClip = preDraw(g, matrix);

		ImageElement[] paths = this.paths;
		for (ImageElement path : paths) {
			path.draw(g, matrix, alpha);
		}

		postDraw(g, imageClip);
	}

	/**
	 * Draws the paths with given matrix at a specific time.
	 *
	 * @param g
	 *            the graphics context to draw on
	 * @param matrix
	 *            the matrix to apply
	 * @param elapsedTime
	 *            the time elapsed within the overall animation, in milliseconds
	 */
	@Override
	/* package */ void drawAnimated(GraphicsContext g, Matrix matrix, long elapsedTime) {
		checkClosed();
		ImageClip imageClip = preDraw(g, matrix);

		ImageElement[] paths = this.paths;
		for (ImageElement path : paths) {
			Matrix localMatrix = new Matrix(matrix);
			path.drawAnimated(g, localMatrix, elapsedTime);
		}

		postDraw(g, imageClip);
	}

	/**
	 * Draws the paths with given matrix at a specific time.
	 *
	 * @param g
	 *            the graphics context to draw on
	 * @param matrix
	 *            the matrix to apply
	 * @param alpha
	 *            the global opacity rendering value
	 * @param elapsedTime
	 *            the time elapsed within the overall animation, in milliseconds
	 */
	@Override
	/* package */ void drawAnimated(GraphicsContext g, Matrix matrix, int alpha, long elapsedTime) {
		checkClosed();
		checkOverlapAlpha(alpha);
		ImageClip imageClip = preDraw(g, matrix);

		ImageElement[] paths = this.paths;
		for (ImageElement path : paths) {
			Matrix localMatrix = new Matrix(matrix);
			path.drawAnimated(g, localMatrix, alpha, elapsedTime);
		}

		postDraw(g, imageClip);
	}

	/**
	 * Draws the paths with given matrix and with colors transformed with the given matrix.
	 *
	 * @param g
	 *            the graphics context to draw on
	 * @param matrix
	 *            the matrix to apply
	 * @param colorMatrix
	 *            the color matrix to apply
	 */
	@Override
	public void drawFiltered(GraphicsContext g, Matrix matrix, float[] colorMatrix) {
		checkClosed();
		checkOverlapAlpha(colorMatrix);
		ImageClip imageClip = preDraw(g, matrix);

		ColorMatrixTransformer transformer = new ColorMatrixTransformer(colorMatrix);

		ImageElement[] paths = this.paths;
		for (ImageElement path : paths) {
			path.drawTransformed(g, matrix, transformer);
		}

		postDraw(g, imageClip);
	}

	/**
	 * Draws the paths with given matrix at a specific time and with colors transformed with the given matrix.
	 *
	 * @param g
	 *            the graphics context to draw on
	 * @param matrix
	 *            the matrix to apply
	 * @param colorMatrix
	 *            the color matrix to apply
	 * @param elapsedTime
	 *            the time elapsed within the overall animation, in milliseconds
	 */
	@Override
	public void drawFilteredAnimated(GraphicsContext g, Matrix matrix, long elapsedTime, float[] colorMatrix) {
		checkClosed();
		checkOverlapAlpha(colorMatrix);
		ImageClip imageClip = preDraw(g, matrix);

		ColorMatrixTransformer transformer = new ColorMatrixTransformer(colorMatrix);

		ImageElement[] paths = this.paths;
		for (ImageElement path : paths) {
			path.drawTransformedAnimated(g, matrix, elapsedTime, transformer);
		}

		postDraw(g, imageClip);
	}

	private ImageClip preDraw(GraphicsContext g, Matrix matrix) {
		ImageClip imageClip = new ImageClip(g);
		ImageClip.apply(g, matrix, (int) this.getWidth(), (int) this.getHeight());
		return imageClip;
	}

	private void postDraw(GraphicsContext g, ImageClip imageClip) {
		imageClip.revert(g);
	}

	/**
	 * Returns whether this vector image has been closed or not.
	 *
	 * @return <code>true</code> if the vector image has been closed, <code>false</code> otherwise.
	 */
	@Override
	public boolean isClosed() {
		return this.closed;
	}

	/**
	 * Closes this vector image and its associated resources.
	 * <p>
	 * Calling this method on a vector image which has already been closed has no effect.
	 */
	@Override
	public void close() {
		this.closed = true;
	}

}
