/*
 * Java
 *
 * Copyright 2013-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg;

/**
 * Color utilities.
 */
public class ColorHelper {

	/** Constant value for the maximum value that a image component can have. */
	public static final int MAX_COLOR_COMPONENT_VALUE = 0xff;
	/* default */ static final int ALPHA_SHIFT = 24;
	/* default */ static final int RED_SHIFT = 16;
	/* default */ static final int GREEN_SHIFT = 8;
	/* default */ static final int BLUE_SHIFT = 0;
	private static final int ALPHA_MASK = MAX_COLOR_COMPONENT_VALUE << ALPHA_SHIFT;
	private static final int RED_MASK = MAX_COLOR_COMPONENT_VALUE << RED_SHIFT;
	private static final int GREEN_MASK = MAX_COLOR_COMPONENT_VALUE << GREEN_SHIFT;
	private static final int BLUE_MASK = MAX_COLOR_COMPONENT_VALUE << BLUE_SHIFT;

	// Prevents initialization.
	private ColorHelper() {
	}

	/**
	 * Gets a color alpha component.
	 *
	 * @param color
	 *                  the color.
	 * @return the alpha component.
	 */
	public static int getAlpha(int color) {
		return (color & ALPHA_MASK) >>> ALPHA_SHIFT;
	}

	/**
	 * Gets a color red component.
	 *
	 * @param color
	 *                  the color.
	 * @return the red component.
	 */
	public static int getRed(int color) {
		return (color & RED_MASK) >>> RED_SHIFT;
	}

	/**
	 * Gets a color green component.
	 *
	 * @param color
	 *                  the color.
	 * @return the green component.
	 */
	public static int getGreen(int color) {
		return (color & GREEN_MASK) >>> GREEN_SHIFT;
	}

	/**
	 * Gets a color blue component.
	 *
	 * @param color
	 *                  the color.
	 * @return the blue component.
	 */
	public static int getBlue(int color) {
		return (color & BLUE_MASK) >>> BLUE_SHIFT;
	}

	/**
	 * Gets the color from red, green and blue components.
	 *
	 * @param alpha
	 *                  the alpha component.
	 * @param red
	 *                  the red component.
	 * @param green
	 *                  the green component.
	 * @param blue
	 *                  the blue component.
	 * @return the color.
	 */
	public static int getColor(int alpha, int red, int green, int blue) {
		// Crop colors
		alpha = Math.min(alpha, MAX_COLOR_COMPONENT_VALUE);
		red = Math.min(red, MAX_COLOR_COMPONENT_VALUE);
		green = Math.min(green, MAX_COLOR_COMPONENT_VALUE);
		blue = Math.min(blue, MAX_COLOR_COMPONENT_VALUE);

		return (alpha << ALPHA_SHIFT) | (red << RED_SHIFT) | (green << GREEN_SHIFT) | (blue << BLUE_SHIFT);
	}

	/**
	 * Updates a color alpha component.
	 *
	 * @param color
	 *                  the color.
	 *
	 * @param alpha
	 *                  the alpha component.
	 *
	 * @return the updated color.
	 *
	 */
	public static int updateAlpha(int color, byte alpha) {
		return (alpha << ALPHA_SHIFT) | (color & (RED_MASK | GREEN_MASK | BLUE_MASK));
	}

}