/*
 * Java
 *
 * Copyright 2022-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg;

import ej.microui.MicroUIException;
import ej.microui.display.BufferedImage;
import ej.microui.display.Format;
import ej.microui.display.GraphicsContext;

/**
 * A buffered vector image is a vector image that can be built dynamically.
 * <p>
 * It reserves the format {@link Format#CUSTOM_7} from MicroUI.
 *
 * @see BufferedImage
 */
public class BufferedVectorImage extends ResourceVectorImage {

	private final BufferedImage bufferedImage;

	/**
	 * Creates a buffered image that holds the GPU commands.
	 * <p>
	 * A command is added in the image when the image's {@link GraphicsContext} is used to draw something into.
	 *
	 * @param w
	 *            the image width
	 * @param h
	 *            the image height
	 *
	 * @see BufferedImage#getGraphicsContext()
	 */
	public BufferedVectorImage(int w, int h) {
		super(w, h);
		this.bufferedImage = new BufferedImage(w, h, Format.CUSTOM_7);
		GraphicsContext graphicsContext = this.bufferedImage.getGraphicsContext();

		VectorGraphicsNatives.mapContext(graphicsContext.getSNIContext(), getSNIContext());
	}

	/**
	 * Returns the {@link GraphicsContext} associated with this image, which may be used in order to draw on the image.
	 * <p>
	 * This method always returns the same {@link GraphicsContext} instance for a specific image. The graphics context
	 * has the same dimensions as the image, which allows to modify every pixel of the image.
	 *
	 * @throws MicroUIException
	 *             if this image has been closed (see {@link #close()}).
	 * @return the image's graphics context.
	 */
	public GraphicsContext getGraphicsContext() {
		checkClosed();
		return this.bufferedImage.getGraphicsContext();
	}

	/**
	 * Clears all commands.
	 */
	public void clear() {
		// use the GC instead of the VG image' sni context to be able to sync the "cleaning" action with the graphics
		// engine
		checkClosed();
		VectorGraphicsNatives.clear(this.bufferedImage.getSNIContext());
	}

	@Override
	public void close() {

		// first: close the BVI
		super.close();

		// here:
		// - the BVI's SNIContext doesn't point anymore on the BI
		// - the BVI cannot be used anymore as source (cannot draw the BVI)
		// - but can draw into the BVI's BI if the BVI's gc has been saved before calling this method

		// then: close the BI (== clear BVI elements + close BI's gc + free image heap)
		this.bufferedImage.close();

		// here:
		// - cannot draw anymore in the BI (cannot be used as target)
		// - BI elements (== BVI elements) have been freed
		// - BI has been removed from the image heap
	}

	@Override
	public ResourceVectorImage filterImage(float[] colorMatrix) {

		// Need to create a new BufferedVectorImage because this bvi can contain some blocks "RAW image" (a block that
		// points on a RAW image). In that case, the content of the RAW image must be drawn in the new BVI to be able to
		// alterate the colors with the color matrix.

		BufferedVectorImage bvi = new BufferedVectorImage((int) getWidth(), (int) getHeight());
		VectorGraphicsPainter.drawFilteredImage(bvi.getGraphicsContext(), this, new Matrix(), colorMatrix);
		return bvi;
	}
}
