/*
 * Java
 *
 * Copyright 2013-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.util;

import ej.annotation.Nullable;

/*
 * Implementation comes from B-ON 1.2
 * 
 */
public abstract class TimerTask implements Runnable {
	
	//value of scheduleTime when the TimerTask has not been scheduled
	private static final int NOT_SCHEDULED = -1;

	//next in the Timer linked list of task
	@Nullable
	TimerTask next;
    //time at which this timer task should be launched (initialized to NOT_SCHEDULED)
    long absoluteTime;
    //period of the task, 0 if the task is not repeatable
    long period;
    //set to true when the task has run or is running
    boolean hasRun;
    //set to true when the task is canceled
    boolean isCanceled;
    boolean fixedRate;

	protected TimerTask() {
		this.absoluteTime = NOT_SCHEDULED;
	}

	public boolean cancel() {
		if(this.isCanceled) {
			return false;
		}

		this.isCanceled = true;
		long originalPeriod = this.period;
		this.period = 0;//avoid reschedule of the task

		//return true if the task is scheduled. If the task is one-time execution and has
		//run, returns false.
		return absoluteTime != NOT_SCHEDULED && !(hasRun && originalPeriod == 0);
	}

	public abstract void run();

	public long scheduledExecutionTime() {
		return this.absoluteTime;
	}
	
	public synchronized void setScheduled(Timer timer, long absoluteTime, long period, boolean fixedRate){

		if (this.absoluteTime != TimerTask.NOT_SCHEDULED) {
			throw new IllegalStateException();//the task has already been scheduled
		}

		this.absoluteTime = absoluteTime;
		this.period = period;
		this.fixedRate = fixedRate;
	}
	
	public void setLaunched(){
		//mark the current task has been launched (ie. it cannot be canceled)
	}
	
	public void uncaughtException(Timer timer, Throwable e){
		this.cancel();
		// this printStackTrace is useful to know the cause of the exception
		e.printStackTrace();//NOSONAR
	}
}
