/*
 * Java
 *
 * Copyright 2013-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.util;

import ej.annotation.Nullable;

/*
 * Implementation comes from B-ON 1.2
 * 
 */
public class Timer {
	
	private class TimerThread implements Runnable {

		// Cyclomatic Complexity : Cyclomatic Complexity is 16 (max allowed is 10)
		// Necessary here
		@Override //NOSONAR
		public void run() {
				
			Object lock = Timer.this.lock;

			// Checks if the timer has not already been started (can be started only once)
			synchronized (lock) {
				if(Timer.this.started) {
					throw new IllegalStateException();
				}
				Timer.this.started = true;
			}

			while(!Timer.this.isCanceled) {
				long currentTime = System.currentTimeMillis();
				// Extract from timer linked list of tasks, a list of tasks to launch now

				TimerTask taskToLaunch = null;// linked list of tasks to launch
				synchronized (lock) {
					// find last task to launch now and first task that should not be launched now
					taskToLaunch = Timer.this.nextTask;
					TimerTask lastTaskToLaunch = null;
					TimerTask firstTaskToNotLaunch = taskToLaunch;
					while(firstTaskToNotLaunch != null && firstTaskToNotLaunch.absoluteTime <= currentTime) {
						lastTaskToLaunch = firstTaskToNotLaunch;
						firstTaskToNotLaunch = firstTaskToNotLaunch.next;
					}
					// here if lastTaskToLaunch is not null, there are tasks to launch now
					if(lastTaskToLaunch != null) {
						// remove all tasks to launch from the scheduled tasks
						Timer.this.nextTask = firstTaskToNotLaunch;
						lastTaskToLaunch.next = null;
						// mark all tasks to launch as launched
						TimerTask task = taskToLaunch;
						while (task != null) {
							task.setLaunched();
							task = task.next;
						}
					} 
					else { // no task to launch now
						   // firstTaskToLaunch is the next task to launch
						try {
							if (taskToLaunch != null) {
								// there is a task scheduled
								// sleep to wake up when next task should be launched
								long sleepTime = taskToLaunch.absoluteTime - System.currentTimeMillis();
								taskToLaunch = null;
								if (sleepTime > 0) {
									lock.wait(sleepTime);
								}
								// else next task should be launched now (continue)
							} else {
								// there is no task scheduled
								lock.wait();// wait infinite
							}
						} catch (InterruptedException e) {
							
						}
						// here we can have be woke up because
						// - task has been added to timer
						// - task should be launched
						// - timer is stopped
						// - current thread has been interrupted
						//
						// continue execution, a new loop will be done unless this pump has been
						// stopped
					}
				}

				// launch all tasks to launch and reschedule it
				// no need to be synchronized here, tasks to launch cannot be added to
				// a timer (i.e. their "next" field cannot be modified) because they are marked as
				// launched
				while (taskToLaunch != null && !isCanceled) {
					TimerTask nextTask = taskToLaunch.next;// next should be read here because it
															// will be updated by reschedule
					if (!taskToLaunch.isCanceled) {
						taskToLaunch.hasRun = true;
						long executionTime = taskToLaunch.fixedRate ? taskToLaunch.absoluteTime : System.currentTimeMillis();
						try {
							taskToLaunch.run();
						} catch (Throwable e) {
							try {
								taskToLaunch.uncaughtException(Timer.this, e);
							} catch (Throwable e2) {
								// nothing to do : ignore all errors
							}
						}
						// schedule the task if it is repeatable (check is done in the timer)
						Timer.this.reschedule(taskToLaunch, executionTime);
					} else {
						// task has been canceled
						taskToLaunch.next = null;
					}
					taskToLaunch = nextTask;
				}
			}
		}
		
	}
	
	// Ordered linked list of scheduled tasks
	@Nullable
	private TimerTask nextTask;
	// Set to true when cancel() method is called
	private boolean isCanceled;
	//set to true when run() method is executed
	private boolean started;

	private Object lock;

	public Timer() {
		this("Timer" + System.currentTimeMillis()); //$NON-NLS-1$
	}
	
	public Timer(String name) {
		this.lock = new Object();
		new Thread(new TimerThread(), name).start();
	}
	
	public void cancel() {
		synchronized (lock) {
			//stop the pump and remove all links to tasks and other objects to allow GC
			this.lock.notify();

			this.isCanceled = true;
			//remove all links between task
			TimerTask task = this.nextTask;
			while(task != null){
				TimerTask nextTask = task.next;
				task.next = null;
				task = nextTask;
			}
		}
	}

	public int purge() {
		synchronized (lock) {
			TimerTask task = this.nextTask;
			TimerTask previous = null;
			int count = 0;
			while(task != null) {
				if(task.isCanceled) {
					if(previous == null) {
						// head of the linked list
						this.nextTask = task.next;
					}
					else {
						previous.next = task.next;
					}
					count++;
				}
				
				previous = task;
				task = task.next;
			}
			this.lock.notify();
			return count;
		}
	}

	public void schedule(TimerTask task, Date time) {
		long absoluteTime = time.getTime();
		if(absoluteTime < 0) {
			throw new IllegalArgumentException();
		}

		this.schedule(task, absoluteTime, 0, false);
	}

	public void schedule(TimerTask task, Date firstTime, long period) {
		long absoluteTime = firstTime.getTime();
		if(absoluteTime < 0 || period <= 0) {
			throw new IllegalArgumentException();
		}

		this.schedule(task, absoluteTime, period, false);
	}

	public void schedule(TimerTask task, long delay) {
		long absoluteTime = System.currentTimeMillis() + delay;
		if (delay < 0 || absoluteTime < 0) {
			throw new IllegalArgumentException();
		}

		this.schedule(task, absoluteTime, 0, false);
	}

	public void schedule(TimerTask task, long delay, long period) {
		long absoluteTime = System.currentTimeMillis() + delay;
		if(delay < 0 || period <= 0 || absoluteTime < 0) {
			throw new IllegalArgumentException();
		}

		this.schedule(task, absoluteTime, period, true);
	}

	public void scheduleAtFixedRate(TimerTask task, Date firstTime, long period) {
		long absoluteTime = firstTime.getTime();
		if(absoluteTime < 0 || period <= 0) {
			throw new IllegalArgumentException();
		}

		this.schedule(task, absoluteTime, period, true);
	}

	public void scheduleAtFixedRate(TimerTask task, long delay, long period) {
		long absoluteTime = System.currentTimeMillis() + delay;
		if(delay < 0 || period <= 0 || absoluteTime < 0) {
			throw new IllegalArgumentException();
		}
		
		this.schedule(task, absoluteTime, period, true);
	}
	
	protected void addTaskToScheduledTasks(TimerTask task, long absoluteTime){
		//Add the task in nextTask linked list.
		//absoluteTime should be equals to task.absoluteTime.
		//This method is not thread safe. Caller should be synchronized on
		//this.lock before calling this method.

		TimerTask previousTask = null;
		TimerTask nextTask = this.nextTask;
		//find previous and next task
		while(nextTask != null && nextTask.absoluteTime < absoluteTime) {
			previousTask = nextTask;
			nextTask = nextTask.next;
		}

		//add it to the list
		task.next = nextTask;

		if(previousTask != null) {
			previousTask.next = task;
		}
		else {
			//timer next task is updated -> wake up pump
			this.nextTask = task;
			this.lock.notify();
		}
	}
	
	protected void reschedule(TimerTask task, long previousExecutionTime){
		//schedule the task if it is repeatable
		//otherwise, remove it from all linked list.
		//This method is thread safe

		//previousExecutionTime value depends on fixedRate attribute of the task:
		//- if the task HAS NOT a fixed rate : it is the computed scheduled execution time of the previous launch
		//- if the task HAS a fixed rate : it is the actual scheduled execution time of the previous launch


		long period = task.period;
		if(period != 0){
			//task is repeatable
			synchronized (lock) {
				//check if timer is not canceled
				if(!this.isCanceled){
					//here the task may have been canceled,
					//it will be scheduled anyway but never launch. It is not a problem.

					//compute its next absolutetime
					long absoluteTime = previousExecutionTime + period;
					task.absoluteTime = absoluteTime;//update task absolute time
					addTaskToScheduledTasks(task,absoluteTime);
				}
			}
		}
		else {
			//task is not repeatable, remove all its links to other task or timer
			task.next = null;
		}
	}
	
	protected void schedule(TimerTask task, long absoluteTime, long period, boolean fixedRate){
		//absoluteTime should be positive,
		//period should be positive (if period == 0 then alarm is not repeatable).
		//This method is thread safe

		//check if task has not already been scheduled and if this timer is running
		synchronized (lock) {

			if(this.isCanceled) {
				throw new IllegalStateException();
			}

			//configure task and check if it has not already been scheduled
			task.setScheduled(this, absoluteTime, period, fixedRate);
			this.addTaskToScheduledTasks(task, absoluteTime);
		}
	}

}
