/*
 * Java 
 * 
 * Copyright 2004-2019 IS2T. All rights reserved. 
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.util;

import java.io.Serializable;

import com.is2t.vm.support.DefaultTimeZoneImpl;
import com.is2t.vm.support.TimeZoneImpl;

import ej.annotation.NonNull;
import ej.annotation.Nullable;

public abstract class TimeZone implements Cloneable, Serializable {

	public static final int LONG = 1;
	public static final int SHORT = 0;

	public static final String TimeZoneProperty = "user.timezone"; //$NON-NLS-1$

	/*
	 * Internal
	 */
	private static final String TimeZoneImplProperty = DefaultTimeZoneImpl.SUPPORT_PREFIX + "timezone.impl"; //$NON-NLS-1$

	/*
	 * Implementation is out of CLDC-1.1 scope.
	 */
	private static TimeZoneImpl impl;

	static {
		// Instanciate TimeZone implementation
		String implClassname = System.getPropertyNoCheck(TimeZoneImplProperty);
		TimeZoneImpl t;
		createImpl: {
			if (implClassname != null) {
				try {
					t = (TimeZoneImpl) Class.forName(implClassname).newInstance();
					break createImpl;
				} catch (Throwable e) {
				}
			}
			t = new DefaultTimeZoneImpl();
		}
		impl = t;
	}

	@Nullable
	protected String id;

	public TimeZone() {
	}

	public Object clone() {
		// code is written in a way it avoids compilation errors within Eclipse 
		// (CloneNotSupportedException not recognized)
		try {
			@SuppressWarnings("null")
			@NonNull TimeZone result = (TimeZone) super.clone();
			return result;
		} catch (Throwable e) {
			// TimeZone implements Cloneable, will not happen
			throw new InternalError();
		}
	}

	// public final String getDisplayName() {
	// }

	// public final String getDisplayName(boolean daylight, int style) {
	// }

	public static synchronized String[] getAvailableIDs() {
		return impl.getAvailableIDs();
	}

	// public static synchronized String[] getAvailableIDs(int rawOffset) {
	// }

	public int getDSTSavings() {
		// Default impl.
		if (this.useDaylightTime()) {
			return 3600000;
		} else {
			return 0;
		}
	}

	public static TimeZone getDefault() {
		return impl.getDefault();
	}

	// public static void setDefault(TimeZone zone) {
	// }

	public String getID() {
		return this.id;
	}

	public static synchronized TimeZone getTimeZone(String id) {
		return impl.getTimeZone(id);
	}

	public abstract int getOffset(int era, int year, int month, int day, int dayOfWeek, int milliseconds);

	public int getOffset(long date) {
		if (this.inDaylightTime(new Date(date))) {
			return this.getRawOffset() + this.getDSTSavings();
		}
		return getRawOffset();
	}

	public abstract int getRawOffset();

	public abstract boolean useDaylightTime();

	public abstract boolean inDaylightTime(Date date);

	public boolean hasSameRules(@Nullable TimeZone other) {
		return other != null && this.getRawOffset() == other.getRawOffset() && this.useDaylightTime() == other.useDaylightTime();
	}

	public void setID(String id) {
		this.id = id;
	}

	public abstract void setRawOffset(int offsetMillis);

}
