/*
 * Java
 *
 * Copyright 2013-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.util;

import ej.annotation.Nullable;

public class Observable {

	private Observer[] observers;
	private int observersPtr;
	private boolean changed;
	
	public Observable() {
		observers = new Observer[4];
		observersPtr = -1;
	}

	public synchronized void addObserver(@Nullable Observer o) {
		if(o == null) {
			throw new NullPointerException(); //NOSONAR // not in spec but consistent with deleteObserver()
		}
		for(int i=observersPtr+1; --i>=0;){
			if(observers[i].equals(o)) {
				return; // already in
			}
		}
		
		try{ observers[++observersPtr] = o; }
		catch(ArrayIndexOutOfBoundsException e){
			System.arraycopy(observers, 0, observers = new Observer[2*observersPtr], 0, observersPtr);
			observers[observersPtr] = o;
		}
	}

	protected synchronized void clearChanged() {
		changed = false;
	}

	public synchronized int countObservers() {
		return observersPtr+1;
	}

	public synchronized void deleteObserver(@Nullable Observer o) {
		if(o == null) {
			return; // spec: passing null has no effect
		}
		for(int i=observersPtr+1; --i>=0;){
			if(observers[i].equals(o)){
				if(i<observersPtr){
					System.arraycopy(observers, i+1, observers, i, observersPtr-i);
				}
				observers[observersPtr] = null;
				--observersPtr;
				return;
			}				
		}
	}

	public synchronized void deleteObservers() {
		// clean-up references
		for(int i=observersPtr+1; --i>=0;){
			observers[i] = null;
		}
		observersPtr = -1;
	}

	public synchronized boolean hasChanged() {
		return changed;
	}

	public void notifyObservers() {
		notifyObservers(null);
	}

	public void notifyObservers(@Nullable Object arg) {
		//See class spec: "[...] The default implementation provided in the Observable class will notify Observers in the order in which they registered interest [...]"
		if(!hasChanged()) {
			return;
		}
		Observer[] tmpObservers;
		int length;
		synchronized (this) {
			// take a snapshot
			length = observersPtr+1;
			tmpObservers = new Observer[length];
			System.arraycopy(observers, 0, tmpObservers, 0, length);
		}

		for(int i=-1; ++i<length;){
			tmpObservers[i].update(this, arg);
		}
		
		clearChanged();
	}

	protected synchronized void setChanged() {
		changed = true;
	}
}
