/*
 * Java
 *
 * Copyright 2004-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.util;

import ej.annotation.NonNull;
import ej.annotation.Nullable;

public class Date implements java.io.Serializable, Cloneable, Comparable<Date> {
	
	private long date;

	public Date() {
		this.date = System.currentTimeMillis();
	}

	public Date(long date) {
		this.date = date;
	}

	public boolean after(Date when) {
		// when.date will throw NullPointerException as specified in Javadoc
		return this.date > when.date;
	}

	public boolean before(Date when) {
		// when.date will throw NullPointerException as specified in Javadoc
		return this.date < when.date;
	}

	@Override
	public Object clone() {
		// code is written in a way it avoids compilation errors within Eclipse 
		// (CloneNotSupportedException not recognized)
		try {
			@SuppressWarnings("null")
			@NonNull Date result = (Date) super.clone();
			return result;
		}
		catch (Throwable e) {
			// Date implements Cloneable so will not happen
			throw new InternalError();
		}
	}

	public int compareTo(Date anotherDate) {
		// anotherDate.date will throw NullPointerException as specified in Javadoc
		// return type is an int, and field "date" is a long. compareTo cannot be
		// implemented like this
		//      return (int) (this.date - anotherDate.date)
		// because of int cast (may result in a bad value)
		if(this.date == anotherDate.date) {
			return 0;
		}
		else if(this.date > anotherDate.date) {
			return 1;
		}
		else {
			return -1;
		}
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		// obj may be null or of another type
		// catch errors and return false
		try {
			return this.date == ((Date) obj).date;
		}
		catch(NullPointerException | ClassCastException e) {
			return false;
		}
	}

	public long getTime() {
		return this.date;
	}

	@Override
	public int hashCode() {
		return (int) (this.date ^ (this.date >>> 32)) ;
	}

	public void setTime(long time) {
		this.date = time;
	}

	@Override
	public String toString() {
		// The result is 28 char long:
		// dow mon dd hh:mm:ss zzz yyyy
		StringBuilder sb = new StringBuilder(28) ;
		Calendar cal = Calendar.getInstance();

    	// make the fileds of Calendar computed
    	cal.setTimeInMillis(this.date);
    	
    	// Day of week
    	sb.append(dayOfWeek(cal.get(Calendar.DAY_OF_WEEK))).append(' ')
    	
    	// Month
   	    .append(month(cal.get(Calendar.MONTH))).append(' ');
    	
    	// Day of month
    	int dom = cal.get(Calendar.DAY_OF_MONTH);
    	if(dom < 10) {
    		sb.append('0');
    	}
    	sb.append(dom).append(' ');
    	
    	// Hour of the day
    	int hour = cal.get(Calendar.HOUR_OF_DAY);
    	if(hour < 10) {
    		sb.append('0');
    	}
    	sb.append(hour).append(':');
    	
    	// Minutes
    	int min = cal.get(Calendar.MINUTE);
    	if(min < 10) {
    		sb.append('0');
    	}
	    sb.append(min).append(':');
	    
    	// Seconds
	    int sec = cal.get(Calendar.SECOND);
    	if (sec < 10) {
    		sb.append('0');
    	}
		sb.append(sec).append(' ');

		TimeZone timeZone = cal.timeZone;
		if(timeZone != null) {
			// Time zone
			sb.append(timeZone.getID()).append(' ');
		}
		
    	// Year
    	sb.append(cal.get(Calendar.YEAR));
   	    
   	    return sb.toString() ;
	}
	
	private static String dayOfWeek(int dow) {
		switch (dow) {
		case Calendar.MONDAY:
			return "Mon"; //$NON-NLS-1$
		case Calendar.TUESDAY:
			return "Tue";//$NON-NLS-1$
		case Calendar.WEDNESDAY:
			return "Wed";//$NON-NLS-1$
		case Calendar.THURSDAY:
			return "Thu";//$NON-NLS-1$
		case Calendar.FRIDAY:
			return "Fri";//$NON-NLS-1$
		case Calendar.SATURDAY:
			return "Sat";//$NON-NLS-1$
		case Calendar.SUNDAY:
			return "Sun";//$NON-NLS-1$
		default:
			return "";//$NON-NLS-1$
		}
	}

	// Cyclomatic Complexity : Cyclomatic Complexity is 13 (max allowed is 10).
	// Not a problem
	private static String month(int m) { //NOSONAR
		switch (m) {
		case Calendar.JANUARY:
			return "Jan";//$NON-NLS-1$
		case Calendar.FEBRUARY:
			return "Feb";//$NON-NLS-1$
		case Calendar.MARCH:
			return "Mar";//$NON-NLS-1$
		case Calendar.APRIL:
			return "Apr";//$NON-NLS-1$
		case Calendar.MAY:
			return "May";//$NON-NLS-1$
		case Calendar.JUNE:
			return "Jun";//$NON-NLS-1$
		case Calendar.JULY:
			return "Jul";//$NON-NLS-1$
		case Calendar.AUGUST:
			return "Aug";//$NON-NLS-1$
		case Calendar.SEPTEMBER:
			return "Sep";//$NON-NLS-1$
		case Calendar.OCTOBER:
			return "Oct";//$NON-NLS-1$
		case Calendar.NOVEMBER:
			return "Nov";//$NON-NLS-1$
		case Calendar.DECEMBER:
			return "Dec";//$NON-NLS-1$
		default:
			return "";//$NON-NLS-1$
		}
	}
}
