/*
 * Java
 *
 * Copyright 2013-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.util;

import java.lang.CloneNotSupportedException;
import java.io.Serializable;

import ej.annotation.NonNullByDefault;
import ej.annotation.Nullable;

@NonNullByDefault(Iterable.DISABLE_NULL_ANALYSIS_COLLECTIONS)
public abstract class AbstractHashMap<K, V>  extends AbstractMap<K, V> implements Map<K, V>, Cloneable, Serializable {

	private static final int DEFAULT_INITIAL_CAPACITY = 16;
	private static final float DEFAULT_LOAD_FACTOR = 0.75f;

	@NonNullByDefault(Iterable.DISABLE_NULL_ANALYSIS_COLLECTIONS)
	protected static abstract class AbstractHashMapEntry<K, V> extends AbstractMapEntry<K, V> implements Cloneable {

		AbstractHashMapEntry<K, V> next; // linked list of entries in buckets

		public AbstractHashMapEntry(V value, AbstractHashMapEntry<K, V> next) {
			this.value = value;
			this.next = next;
		}

		@Override
		@Nullable 
		public V setValue(V value) {
			// value can be null

			V oldValue = this.value;
			this.value = value;
			return oldValue;
		}

		@Override
		protected Object clone() throws CloneNotSupportedException {
			// see Hashtable.clone javadoc
			// Entries are clones, but not keys nor values
			return super.clone();
		}
	}

	// This iterator/enumeration can iterate over three elements:
		// - keys
		// - values
		// - entries
		// It depends what you passed to constructor (KEYS, VALUES or ENTRIES)
	@NonNullByDefault(Iterable.DISABLE_NULL_ANALYSIS_COLLECTIONS)
	/*default*/ class AbstractHashMapEnumeration<E> implements Enumeration<E> {

		int position;
		@Nullable 
		AbstractHashMapEntry<K, V> nextEntry;
		int type;

		public AbstractHashMapEnumeration(int type) {
			this.nextEntry = null;
			this.type = type;
			this.position = 0;
		}

		@Override
		public boolean hasMoreElements() {
			AbstractHashMapEntry<K, V> entry = this.nextEntry;
			int i = this.position;
			// Look for the next entry in the hashtable
			AbstractHashMapEntry<K, V>[] buckets = AbstractHashMap.this.buckets;
			int bucketsLength = buckets.length;
			// Look for the next entry in the hashtable
			while(entry == null && i < bucketsLength) {
				entry = buckets[i++];
			}

			return entry != null;
		}

		@Override
		public E nextElement() {
			if(this.hasMoreElements()) {
				// Look for the next entry in the hashtable
				AbstractHashMapEntry<K, V>[] buckets = AbstractHashMap.this.buckets;
				int bucketsLength = buckets.length;
				AbstractHashMapEntry<K, V> entry = this.nextEntry;
				int position = this.position;
				while(entry == null && position < bucketsLength) {
					entry = buckets[position++];
				}
				this.position = position;

				if(entry != null) {
					setCurrentEntry(entry);
					this.nextEntry = entry.next;
					int type = this.type;
					if(type == KEYS){
						@SuppressWarnings("unchecked")
						E key = (E) entry.getKey();
						return key; // cast is safe for sure
					}
					else if(type == VALUES){
						@SuppressWarnings("unchecked")
						E value = (E) entry.value;
						return value;  // cast is safe for sure
					}
					else {
						//ENTRIES for sure
						@SuppressWarnings("unchecked")
						E result = (E) entry;  // cast is safe for sure
						return result;
					}
				}
				else {
					this.nextEntry = entry;
				}
			}

			throw new NoSuchElementException();
		}

		protected void setCurrentEntry(AbstractHashMapEntry<K, V> entry) {
			// by default, do nothing
		}

	}
	
	@NonNullByDefault(Iterable.DISABLE_NULL_ANALYSIS_COLLECTIONS)
	private class AbstractHashMapIterator<E> extends AbstractHashMapEnumeration<E> implements Iterator<E> {

		int neededmodCount;
		
		@Nullable
		AbstractHashMapEntry<K, V> currentEntry;

		/**
		 * @param type : <code>KEYS</code>, <code>VALUES</code> or <code>ENTRIES</code>
		 */
		public AbstractHashMapIterator(int type) {
			super(type);
			this.neededmodCount = AbstractHashMap.this.modCount;
		}

		@Override
		protected void setCurrentEntry(AbstractHashMapEntry<K, V> entry) {
			currentEntry = entry;
		}

		@Override
		public boolean hasNext() {
			return hasMoreElements();
		}

		@Override
		public E next() {
			if (this.neededmodCount != AbstractHashMap.this.modCount) {
				throw new ConcurrentModificationException();
			}

			return nextElement();
		}

		@Override
		public void remove() {
			AbstractHashMap<K, V> thisHashtable = AbstractHashMap.this;
			if (this.neededmodCount != thisHashtable.modCount) {
				throw new ConcurrentModificationException();
			}
			AbstractHashMapEntry<K, V> currentEntry = this.currentEntry;
			if(currentEntry == null) {
				// Already done once, or 0 call to next()
				throw new IllegalStateException();
			}

			AbstractHashMapEntry<K, V>[] buckets = thisHashtable.buckets;

			int index = thisHashtable.hash(currentEntry.getInternKey());
			AbstractHashMapEntry<K, V> entryPoint = buckets[index];
			AbstractHashMapEntry<K, V> previous = null;

			while(entryPoint != null) {
				if(entryPoint == currentEntry) {
					// update linked list
					if(previous == null) {
						// this.lastElem is the bucket entry point
						buckets[index] = entryPoint.next;
					}
					else {
						previous.next = entryPoint.next;
					}

					this.currentEntry = null;
					this.neededmodCount++; // not a concurrent modification
					thisHashtable.size--;
					thisHashtable.modCount++;

					return;
				}

				previous = entryPoint;
				entryPoint = entryPoint.next;
			}

			// Unable to find in the bucket the current element
			// There was a concurrent modification
			throw new ConcurrentModificationException();
		}
	}

	@NonNullByDefault(Iterable.DISABLE_NULL_ANALYSIS_COLLECTIONS)
	private class AbstractHashMapCollection<E> extends AbstractCollection<E> {

		int type;

		/**
		 * @param type :
		 * 		<code>KEYS</code>, <code>VALUES</code>
		 * 		or <code>ENTRIES</code>
		 */
		public AbstractHashMapCollection(int type) {
			this.type = type;
		}

		@Override
		public boolean equals(@Nullable Object o) {
			if(this == o) {
				return true;
			}

			if(!(o instanceof Collection)) {
				return false;
			}

			Collection<?> oAsCollection = (Collection<?>) o;

			if(this.size() != oAsCollection.size()) {
				return false;
			}

			try {
				return this.containsAll(oAsCollection);
			}
			catch(ClassCastException e) {
				// Set of another type of this
				return false;
			}
		}

		@Override
		public boolean add(E e) {
			// See javadoc
			throw new UnsupportedOperationException();
		}

		@Override
		public boolean addAll(Collection<? extends E> c) {
			// See javadoc
			throw new UnsupportedOperationException();
		}

		@Override
		public void clear() {
			AbstractHashMap.this.clear();
		}

		@Override
		public boolean contains(Object o) {
			int type = this.type;
			if(type == KEYS){
				return AbstractHashMap.this.containsKey(o);
			}
			else if(type == VALUES){
				return AbstractHashMap.this.containsValue(o);
			}
			else {
				//ENTRIES for sure
				return this.containsEntry(o);
			}
		}

		private boolean containsEntry(Object o) {
			if (!(o instanceof Map.Entry)) {
				return false;
			}

			AbstractHashMapEnumeration<Map.Entry<K, V>> enumeration = new AbstractHashMapEnumeration<Map.Entry<K, V>>(ENTRIES);
			while(enumeration.hasMoreElements()) {
				if (enumeration.nextElement().equals(o)) {
					return true;
				}
			}

			return false;
		}

		@Override
		public boolean isEmpty() {
			return AbstractHashMap.this.isEmpty();
		}

		@Override
		public Iterator<E> iterator() {
			// Cannot create the iterator directly, because of LinkedHashMap in Eclasspath
			return AbstractHashMap.this.iterator(this.type);
		}

		@Override
		public boolean remove(Object o) {
			int type = this.type;
			if(type == KEYS){
				return AbstractHashMap.this.remove(o) != null;
			}
			else if(type == ENTRIES){
				return this.removeEntry(o);
			}
			else {
				throw new UnsupportedOperationException();
			}
		}

		private boolean removeEntry(Object o) {
			if (!(o instanceof AbstractHashMapEntry)) {
				return false;
			}

			AbstractHashMap<K, V> thisAbstractHashMap = AbstractHashMap.this;
			AbstractHashMapEntry<K, V>[] buckets = thisAbstractHashMap.buckets;
			int bucketsLength = buckets.length;

			@SuppressWarnings("unchecked")
			AbstractHashMapEntry<K, V> oAsEntry = (AbstractHashMapEntry<K, V>) o; // cast is safe
			for(int i = 0; i < bucketsLength; i++) {
				AbstractHashMapEntry<K, V> entryPoint = buckets[i];
				AbstractHashMapEntry<K, V> previous = null;
				while(entryPoint != null) {
					if(entryPoint.equals(oAsEntry)) {
						if(previous == null) {
							buckets[i] = entryPoint.next;
						}
						else {
							previous.next = entryPoint.next;
						}

						thisAbstractHashMap.size--;
						thisAbstractHashMap.modCount++;

						return true;
					}

					previous = entryPoint;
					entryPoint = entryPoint.next;
				}
			}

			return false;
		}

		@Override
		public int size() {
			return AbstractHashMap.this.size;
		}

	}
	
	@NonNullByDefault(Iterable.DISABLE_NULL_ANALYSIS_COLLECTIONS)
	private final class AbstractHashMapSet<E> extends AbstractHashMapCollection<E> implements Set<E> {

		/**
		 * @param type :
		 * 		<code>KEYS</code> or <code>ENTRIES</code>
		 */
		public AbstractHashMapSet(int type) {
			super(type);

			// If type is VALUES, throw an internal error since values are not
			// unique. VALUES can only be used with AbstractHashMapCollection

			if(this.type == VALUES) {
				throw new InternalError();
			}
		}

		@Override
		public int hashCode() {
			int h = 0;
			Iterator<E> i = iterator();
			while (i.hasNext()) {
				E obj = i.next();
				if (obj != null) {
					h += obj.hashCode();
				}
				}
			return h;
		}
	}

	AbstractHashMapEntry<K, V>[] buckets;
	int size;

	int threshold;
	float loadFactor;
	int modCount; // structural modifications count (same as List)

	public AbstractHashMap() {
		this(DEFAULT_INITIAL_CAPACITY, DEFAULT_LOAD_FACTOR);
	}

	public AbstractHashMap(int initialCapacity) {
		this(initialCapacity, DEFAULT_LOAD_FACTOR);
	}

	public AbstractHashMap(int initialCapacity, float loadFactor) {
		if(initialCapacity < 0) {
			throw new IllegalArgumentException(String.valueOf(initialCapacity));
		}
		if(Float.isInfinite(loadFactor) || Float.isNaN(loadFactor) || loadFactor <= 0) {
			throw new IllegalArgumentException(String.valueOf(loadFactor));
		}

		if(initialCapacity == 0) {
			initialCapacity = 1; // at least one element can be added without incresing capacity
		}
		@SuppressWarnings("unchecked")
		AbstractHashMapEntry<K,V>[] buckets = new AbstractHashMapEntry[initialCapacity];
		this.buckets = buckets; // for sure
		this.loadFactor = loadFactor;
		this.threshold = (int) (loadFactor * initialCapacity);
		init();
	}


	public AbstractHashMap(Map<? extends K, ? extends V> m) {
		// double size to avoid capacity increment cost for the next put
		this(m.size() * 2, DEFAULT_LOAD_FACTOR);
		this.putAll(m);
	}

	/**
	 * Called by all constructors before adding any element.
	 * This method can be overridden by subclasses.
	 */
	void init(){

	}

	@Override
	public void clear() {
		AbstractHashMapEntry<K, V>[] buckets = this.buckets;
		int bucketsLength = buckets.length;
		for(int i = 0; i < bucketsLength; i++) {
			buckets[i] = null;
		}
		this.size = 0;
		this.modCount++;
	}

	@Override
	public boolean containsKey(Object key) {
		int index = this.hash(key);
		AbstractHashMapEntry<K, V> entryPoint = this.buckets[index];

		while(entryPoint != null) {
			if (equalsEntry(key, entryPoint)) {
				return true;
			}
			entryPoint = entryPoint.next;
		}

		// key not found
		return false;
	}

	/**
	 * Tell whether the given entry is the one that holds the given key.
	 * @param key may be null
	 */
	protected boolean equalsEntry(Object key, AbstractHashMapEntry<K, V> entryPoint) {
		if(key == null){
			return entryPoint.getKey() == null;
		}
		else{
			Object entryKey = entryPoint.getInternKey();
			return entryKey != null && entryKey.equals(key);
		}
	}

	@Override
	public boolean containsValue(Object value) {
		// value can be null
		AbstractHashMapEntry<K, V>[] buckets = this.buckets;
		int bucketsLength = buckets.length;
		for(int i = 0; i < bucketsLength; i++) {
			AbstractHashMapEntry<K, V> e = buckets[i];
			while(e != null) {
				if(value == null ? e.value == null : value.equals(e.value)) {
					return true;
				}
				e = e.next;
			}
		}
		return false;
	}

	@Override
	public Set<Map.Entry<K, V>> entrySet() {
		return new AbstractHashMapSet<Map.Entry<K, V>>(ENTRIES);
	}

	@Override
	@Nullable 
	public V get(Object key) {
		int index = this.hash(key);
		AbstractHashMapEntry<K, V> entryPoint = this.buckets[index];

		while(entryPoint != null) {
			if (equalsEntry(key, entryPoint)) {
				// may return null if the value is null and map allows null value
				return entryPoint.value;
			}
			entryPoint = entryPoint.next;
		}

		// key not found
		return null;
	}
	
	@Nullable 
	AbstractHashMapEntry<K, V> getEntry(Object key) {
		int index = this.hash(key);
		AbstractHashMapEntry<K, V> entryPoint = this.buckets[index];

		while(entryPoint != null) {
			if (equalsEntry(key, entryPoint)) {
				return entryPoint;
			}
			entryPoint = entryPoint.next;
		}

		// key not found
		return null;
	}

	@Override
	public boolean isEmpty() {
		return this.size == 0;
	}

	<T> Iterator<T> iterator(int type) {
		return new AbstractHashMapIterator<T>(type);
	}

	@Override
	public Set<K> keySet() {
		Set<K> keys = this.keys;
		if(keys == null){
			this.keys = keys = new AbstractHashMapSet<K>(KEYS);
		}
		return keys;

	}

	@Override
	@Nullable 
	public V put(K key, V value) {
		// value can be null
		int bucketIndex = this.hash(key);
		// 2 scenarii:
		// - the key is already in the table: update the value, return the old value
		// - the key does not exists: add the entry, return null (null is the "old value")

		AbstractHashMapEntry<K, V> entryPoint = this.buckets[bucketIndex];
		while(entryPoint != null) {
			if (equalsEntry(key, entryPoint)) {
				// first scenario
				// here, this is an update, not a structural modification
				// do not increment modCount

				// Method call necessary for LinkedHashMap to work correctly in Eclasspath
				entryPoint.access();
				V oldValue = entryPoint.value;
				entryPoint.value = value;
				return oldValue;
			}
			entryPoint = entryPoint.next;
		}

		// second scenario
		// if the table has reached its threshold, we need to rehash the whole table
		// do it before inserting new entry (so there one less entry to move, and because it
		// has to be done now too)
		if(this.size >= this.threshold) {
			this.rehash();
			// do not forget to update bucketIndex since table length has changed!
			bucketIndex = this.hash(key);
		}

		this.modCount++;
		this.size++;
		this.addEntry(key, value, bucketIndex, true);

		return null;
	}

	/*
	 * Helper method for put, that creates and adds a new Entry.  This is
     * overridden in LinkedHashMap for bookkeeping purposes.
     * This is also overridden by WeakHashMap and Hashtable.
	 */
	void addEntry(K key, V value, int idx, boolean callRemove) {
		AbstractHashMapEntry<K, V> entryPoint = this.buckets[idx];
		// insert new entry at beginning to avoid an iteration over entry linked list
		// this is not specified in documentation
		AbstractHashMapEntry<K, V> newEntry = newHashEntry(key, value, entryPoint);
		this.buckets[idx] = newEntry;
	}

	protected abstract AbstractHashMapEntry<K, V> newHashEntry(K key, V value, AbstractHashMapEntry<K, V> entryPoint);

	@Override
	public void putAll(Map<? extends K, ? extends V> m) {
		// t.entrySet() will throw NullPointerException as expected in javadoc
		for(Map.Entry<? extends K, ? extends V> entry : m.entrySet()) {
			this.put(entry.getKey(), entry.getValue());
		}
	}

	protected void rehash() {
		// Steps :
		// - increase table capacity
		// - move elements in table (= rehash)
		// - update fields to match the new table
		int newCapacity = this.buckets.length * 2 + 2; // +2 to deal with "0 capacity" table
		@SuppressWarnings("rawtypes")
		AbstractHashMapEntry[] oldData = this.buckets;

		// step 1
		@SuppressWarnings("unchecked")
		AbstractHashMapEntry<K,V>[] newData = new AbstractHashMapEntry[newCapacity];
		this.buckets = newData;

		// step 2
		int oldDataLength = oldData.length;
		for(int i = 0; i < oldDataLength; i++) {
			@SuppressWarnings("unchecked")
			AbstractHashMapEntry<K, V> entryPoint = oldData[i];
			while(entryPoint != null) {
				// use the local element since we need to edit the linked list
				// so we use entryPoint to iterator over the linked list, and
				// we work on element
				AbstractHashMapEntry<K, V> element = entryPoint;
				entryPoint = entryPoint.next;

				// use entryPoint.hashCode (field) = key hashCode
				int bucketIndex = this.hash(element.getInternKey());
				// insert at beginning (easiest, fastest)
				element.next = newData[bucketIndex];
				newData[bucketIndex] = element;
			}
		}

		// step 3
		this.modCount++;
		this.threshold = (int) (newCapacity * loadFactor); // next rehash
	}

	@Override
	@Nullable 
	public V remove(Object key) {
		int bucketIndex = this.hash(key);

		AbstractHashMapEntry<K, V> entryPoint = this.buckets[bucketIndex];
		AbstractHashMapEntry<K, V> previous = null;
		while(entryPoint != null) {
			if (equalsEntry(key, entryPoint)) {
				if(previous == null) {
					this.buckets[bucketIndex] = entryPoint.next;
				}
				else {
					previous.next = entryPoint.next;
				}

				this.modCount++;
				this.size--;
				// Method call necessary for LinkedHashMap to work correctly in Eclasspath
				return entryPoint.cleanup();
			}

			previous = entryPoint;
			entryPoint = entryPoint.next;
		}

		// Key does not exists.
		return null;
	}

	@Override
	public int size() {
		return this.size;
	}

	@Override
	public Collection<V> values() {
		// Since AbstractHashMapSet is just a wrapper over AbstractHashMap,
		// we can use it as a Collection over values (AbstractHashMapSet does
		// not support add operations, so the "Set contract" has not to be followed)
		if(values == null){
			values = new AbstractHashMapCollection<V>(VALUES);
		}
		return values;

	}

	protected int hash(Object key) {
		return key == null ? 0 : Math.abs(key.hashCode() % buckets.length);
	}

	/*default*/ static<K,V> void cloneEntries(AbstractHashMapEntry<K, V>[] orig, AbstractHashMapEntry<K, V>[] clone) throws CloneNotSupportedException{
		int length = orig.length;
		for(int i = length; --i >=0 ;) {
			AbstractHashMapEntry<K, V> bucket = orig[i];
			if(bucket != null){
				//update root of the linked list
				@SuppressWarnings("unchecked")
				AbstractHashMapEntry<K, V> cloneBucketRoot = (AbstractHashMapEntry<K, V>) bucket.clone();
				bucket = cloneBucketRoot;
				clone[i] = bucket;

				//copy the linked list
				AbstractHashMapEntry<K, V> previousBucket = bucket;
				bucket = bucket.next;
				while(bucket != null) {
					@SuppressWarnings("unchecked")
					AbstractHashMapEntry<K, V> cloneBucket = (AbstractHashMapEntry<K, V>) bucket.clone();
					bucket = cloneBucket;
					previousBucket.next = bucket;
					previousBucket = bucket;
					bucket = bucket.next;
				}
			}
		}
	}
}