/*
 * Java
 *
 * Copyright 2013-2019 IS2T. All rights reserved
 * IS2T PROPRIETARY/CONFIDENTIAL Use is subject to license terms.
 */
package java.util;

import com.is2t.tools.ArrayTools;

import ej.annotation.NonNullByDefault;

@NonNullByDefault(Iterable.DISABLE_NULL_ANALYSIS_COLLECTIONS)
public abstract class AbstractCollection<E> implements Collection<E> {

	protected AbstractCollection() {
	}

	@Override
	public boolean add(E e) {
		// In this abstract class, we don't know how to represent
		// the content of the collection. So we can't impl. add here.
		// Must be done in subclasses

		// See javadoc for thrown exception
		throw new UnsupportedOperationException();
	}

	@Override
	public boolean addAll(Collection<? extends E> c) {
		boolean hasBeenModified = false;
		for(E elem : c) {
			boolean res = this.add(elem);
			hasBeenModified = res || hasBeenModified;
		}
		return hasBeenModified;
	}

	@Override
	public void clear() {
		// Generic implementation
		// Can be override in subclasses to be more efficient
		Iterator<E> iterator = this.iterator();
		while(iterator.hasNext()) {
			iterator.next();
			iterator.remove();
		}
	}

	@Override
	public boolean contains(Object o) {
		// Generic implementation
		// Can be override in subclasses to be more efficient
		boolean oIsNull = (o == null);
		Iterator<E> iterator = this.iterator();
		while(iterator.hasNext()) {
			E elem = iterator.next();
			// o is in this collection if:
			//    - o is null and there is a null in this collection
			//    - o is not null and o is equal to another element
			if(oIsNull) {
				if(elem == null) {
					return true;
				}
			}
			else {
				assert o != null;
				if(o.equals(elem)) {
					return true;
				}
			}
		}
		return false;
	}

	@Override
	public boolean containsAll(Collection<?> c) {
		for(Object elem : c) {
			if(!this.contains(elem)) {
				return false;
			}
		}
		return true;
	}

	@Override
	public boolean isEmpty() {
		// Generic implementation
		// Can be override in subclasses to be more efficient
		return this.size() == 0;
	}

	@Override
	public abstract Iterator<E> iterator();

	@Override
	public boolean remove(Object o) {
		// Generic implementation
		// Can be override in subclasses to be more efficient
		boolean oIsNull = (o == null);
		Iterator<E> iterator = this.iterator();
		while(iterator.hasNext()) {
			E elem = iterator.next();
			// An element e is removed from this collection if:
			//    - e is null and o is null too
			//    - e is not null and e is equal to o
			boolean found = false;
			if(oIsNull) {
				found = elem == null;
			}
			else {
				assert o != null;
				found = o.equals(elem);
			}
			
			if(found) {
				iterator.remove();
				// Remove only one element. So return immediatly
				return true;
			}
		}
		return false;
	}

	@Override
	public boolean removeAll(Collection<?> c) {
		// If you modify this code, modify retainAll(Collection<?>) too
		if (c == null) {
			throw new NullPointerException();
		}

		boolean hasBeenModified = false;
		Iterator<E> iterator = this.iterator();
		while(iterator.hasNext()) {
			if(c.contains(iterator.next())) {
				iterator.remove();
				hasBeenModified = true;
			}
		}
		return hasBeenModified;
	}

	@Override
	public boolean retainAll(Collection<?> c) {
		// If you modify this code, modify removeAll(Collection<?>) too
		if (c == null) {
			throw new NullPointerException();
		}

		boolean hasBeenModified = false;
		Iterator<E> iterator = this.iterator();
		while(iterator.hasNext()) {
			if(!c.contains(iterator.next())) {
				iterator.remove();
				hasBeenModified = true;
			}
		}
		return hasBeenModified;
	}

	@Override
	public abstract int size();

	@Override
	public Object[] toArray() {
		int thisSize = this.size();
		Object[] array = new Object[thisSize];
		Iterator<E> iterator = this.iterator();
		int i=0;
		while(iterator.hasNext()) {
			try {
				array[i++] = iterator.next();
			}
			catch(ArrayIndexOutOfBoundsException e) {
				// implementation does not seem to throw ConcurrentModificationException
				// Iterator impl. has more objects than the expected size (this is not mandatory but
				// throw ConcurrentModificationException to help debug)
				throw new ConcurrentModificationException();
			}
		}

		if(i < thisSize) {
			// Same reasons
			throw new ConcurrentModificationException();
		}

		return array;
	}

	@SuppressWarnings("unchecked")
	@Override
	public <T> T[] toArray(T[] a) {
		int thisSize = this.size();
		int destSize = a.length;

		if(thisSize > destSize) {
			a = (T[])ArrayTools.createNewArrayFromType(a.getClass(), thisSize);
		}

		Iterator<E> iterator = this.iterator();
		int i=0;
		while(iterator.hasNext()) {
			try {
				a[i++] = (T) iterator.next();
			}
			catch(ArrayIndexOutOfBoundsException e) {
				// implementation does not seem to throw ConcurrentModificationException
				// Iterator impl. has more objects than the expected size (this is not mandatory but
				// throw ConcurrentModificationException to help debug)
				throw new ConcurrentModificationException();
			}
		}

		if(i < thisSize) {
			// Same reasons
			throw new ConcurrentModificationException();
		}
		if(destSize > thisSize){
			a[thisSize] = null; //null-terminate
		}
		return a;
	}

	@Override
	public String toString() {
		StringBuilder b = new StringBuilder().append('[');
		Iterator<E> iterator = this.iterator();
		while(iterator.hasNext()) {
			E elem = iterator.next();
			// Check if the element is "this" collection (to ensure no infinite recursion call of toString)
			b.append(elem == this ? "this" : elem); //$NON-NLS-1$
			if(iterator.hasNext()) {
				b.append(", "); //$NON-NLS-1$
			}
		}
		b.append(']');
		return b.toString();
	}

}
