/*
 * Java
 *
 * Copyright 2013-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.lang.ref;

import ej.annotation.Nullable;

/**
 * Reference queues, to which registered reference objects are appended by the garbage collector
 * after the appropriate reachability changes are detected.
 */
public class ReferenceQueue<T> {

	@Nullable
	private Reference<T> head;//VM known, linked list
	//we have to use a special object to synchronize on the queue
	//since we do not want to interact with the application synchro
	private Object monitor;//VM known
	
	/**
	 * Constructs a new reference-object queue.
	 */
	public ReferenceQueue() {
		//head = null; VM done
		monitor = new Object();
	}

	/**
	 * Polls this queue to see if a reference object is available. If one is available without
	 * further delay then it is removed from the queue and returned. Otherwise this method
	 * immediately returns <tt>null</tt>.
	 * 
	 * @return A reference object, if one was immediately available, otherwise <code>null</code>
	 */
	@Nullable 
	public Reference<? extends T> poll() {
		Reference<T> result;

		synchronized(monitor)
		{
			result = this.head;
			if(result != null){
				//do not use try{}catch{} here since this code is synchronized with the GC
				// => no allocation must occur here (runtime exception)
				this.head = result.nextInQueue;
				result.isInQueue = false;
			}
		}

		return result;
	}

	/**
	 * Removes the next reference object in this queue, blocking until one becomes available.
	 * 
	 * @return A reference object, blocking until one becomes available
	 * @throws InterruptedException
	 *         If the wait is interrupted
	 */
	public Reference<? extends T> remove() throws InterruptedException {
		Reference<? extends T> result = remove(0);
		assert result != null;
		return result;
	}

	/**
	 * Removes the next reference object in this queue, blocking until either one becomes available
	 * or the given timeout period expires.
	 * 
	 * <p>
	 * This method does not offer real-time guarantees: It schedules the timeout as if by invoking
	 * the {@link Object#wait(long)} method.
	 * 
	 * @param timeout
	 *        If positive, block for up to <code>timeout</code> milliseconds while waiting for a
	 *        reference to be added to this queue. If zero, block indefinitely.
	 * 
	 * @return A reference object, if one was available within the specified timeout period,
	 *         otherwise <code>null</code>
	 * 
	 * @throws IllegalArgumentException
	 *         If the value of the timeout argument is negative
	 * 
	 * @throws InterruptedException
	 *         If the timeout wait is interrupted
	 */
	@Nullable 
	public Reference<? extends T> remove(long timeout) throws IllegalArgumentException, InterruptedException {
		Reference<? extends T> result;
		synchronized(monitor)
		{
			if(timeout == 0){
				//wait until data is available
				while((result=poll()) == null){
					monitor.wait();
					//We have been waken up because an element has been added in the queue but
					//somebody may catch it before us (with a call to poll()) so we need to
					//loop until some data is available.
				}
			}
			else {
				//wait until data is available or timeout
				result = poll();
				if(result == null){
					monitor.wait(timeout);
					result = poll();
				}
			}
		}
		return result;
	}
	

	// return false if the reference is already enqueued or not registered to a queue, true otherwise
	static <T> boolean enqueue(Reference<T> reference){
		ReferenceQueue<T> queue = reference.queue;
		if(queue != null){
			return queue.enqueueIntern(reference);
		}
		else {
			return false;
		}
	}
		
	private boolean enqueueIntern(Reference<T> reference){
		synchronized (monitor) {
			//note: check again that reference has a queue because it may have been modified before previous check and synchronize
			if(!reference.isInQueue && reference.queue!=null){
				Reference<T> oldHead = this.head;
				reference.nextInQueue = oldHead;
				this.head = reference;
				reference.queue = null;
				reference.isInQueue = true;
				
				if(oldHead == null){
	    			//This is the first element added in the refqueue,
					//notify potential waiting thread
					monitor.notify();
				}
				return true;
			}
			else {
				return false;
			}
		}
	}

}
