/*
 * Java
 *
 * Copyright 2008-2024 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.lang;

import java.io.OutputStream;
import java.io.PrintStream;
import java.util.PropertyPermission;

import com.is2t.vm.support.CalibrationConstants;

import ej.annotation.Nullable;

/** IST-API
 * The System class contains several useful class fields and methods. It cannot be instantiated.
 */
public final class System extends Object {

	private static final String SYSTEM_OUT_CLASS_PROPERTY = "com.is2t.system.out.class" ;

	// Put LineSep here to avoid cycle between System and PrintStream
	public static final String LineSep;
	private static final String StackFrameStart = "     at ";
	private static final String StackFrameMethodStart = ".";
	private static final String StackFrameIPStart = "(";
	private static final String StackFrameEnd = ")\n";
	private static final String RemainingFrameStart = "...\n";
	private static final String RemainingFrameEnd = " remaining frame(s).\n";


	/** IST-API
	 * The "standard" output stream. This stream is already open and ready to
	 * accept output data. Typically this stream corresponds to display output
	 * or another output destination specified by the host environment or user.
	 *
	 * For simple stand-alone Java applications, a typical way to write a line
	 * of output data is:
	 *
	 *                System.out.println(data)
	 *
	 * See the println methods in class PrintStream.
	 */
	public static final PrintStream out;

	public native static OutputStream getOutputStream();
	
	static {
		java.io.OutputStream tmp = buildOutputStream();
		out = new java.io.PrintStream(tmp);
		// System_out.println(out);
		String lineSep = getProperty("line.separator", "\n");
		assert lineSep != null;
		LineSep = lineSep;
	}

	/** IST-API */
	private System() {
		//CANNOT BE INSTANTIATED
	}

	/** IST-API
	 * The "standard" error output stream. This stream is already open and ready to
	 * accept output data.
	 *
	 * Typically this stream corresponds to display output or another output
	 * destination specified by the host environment or user. By convention, this
	 * output stream is used to display error messages or other information that
	 * should come to the immediate attention of a user even if the principal
	 * output stream, the value of the variable out, has been redirected to a file
	 * or other destination that is typically not continuously monitored.
	 */
	public static final PrintStream err = out ;


	/** IST-API
	 * Copies an array from the specified source array, beginning at the
	 * specified position, to the specified position of the destination array.
	 * A subsequence of array components are copied from the source array
	 * referenced by src to the destination array referenced by dst. The number
	 * of components copied is equal to the length argument. The components at
	 * positions srcOffset through srcOffset+length-1 in the source array are
	 * copied into positions dstOffset through dstOffset+length-1,
	 * respectively, of the destination array.
	 *
	 * If the src and dst arguments refer to the same array object, then the
	 * copying is performed as if the components at positions srcOffset through
	 * srcOffset+length-1 were first copied to a temporary array with length
	 * components and then the contents of the temporary array were copied into
	 * positions dstOffset through dstOffset+length-1 of the destination array.
	 *
	 * If dst is null, then a NullPointerException is thrown.
	 *
	 * If src is null, then a NullPointerException is thrown and the
	 * destination array is not modified.
	 *
	 * Otherwise, if any of the following is true, an ArrayStoreException is
	 * thrown and the destination is not modified: The src argument refers to
	 * an object that is not an array. The dst argument refers to an object
	 * that is not an array. The src argument and dst argument refer to arrays
	 * whose component types are different primitive types. The src argument
	 * refers to an array with a primitive component type and the dst argument
	 * refers to an array with a reference component type. The src argument
	 * refers to an array with a reference component type and the dst argument
	 * refers to an array with a primitive component type.
	 *
	 * Otherwise, if any of the following is true, an IndexOutOfBoundsException
	 * is thrown and the destination is not modified: The srcOffset argument is
	 * negative. The dstOffset argument is negative. The length argument is
	 * negative. srcOffset+length is greater than src.length, the length of the
	 * source array. dstOffset+length is greater than dst.length, the length of
	 * the destination array.
	 *
	 * Otherwise, if any actual component of the source array from position
	 * srcOffset through srcOffset+length-1 cannot be converted to the
	 * component type of the destination array by assignment conversion, an
	 * ArrayStoreException is thrown. In this case, let k be the smallest
	 * nonnegative integer less than length such that src[srcOffset+k] cannot
	 * be converted to the component type of the destination array; when the
	 * exception is thrown, source array components from positions srcOffset
	 * through srcOffset+k-1 will already have been copied to destination array
	 * positions dstOffset through dstOffset+k-1 and no other positions of the
	 * destination array will have been modified. (Because of the restrictions
	 * already itemized, this paragraph effectively applies only to the
	 * situation where both arrays have component types that are reference
	 * types.)
	 */
	native public static void arraycopy(Object src, int srcOffset, Object dst,
			int dstOffset, int length) ;

	/** IST-API
	 * Returns the current time in milliseconds.
	 */
	native public static long currentTimeMillis();

	/** IST-API
	 * <p>Terminates the currently running Java application. The argument serves
	 * as a status code; by convention, a nonzero status code indicates
	 * abnormal termination.</p>
	 *
	 * <p>This method calls the exit method in class Runtime. This method never
	 * returns normally.</p>
	 * <p>
	 * The call System.exit(n) is effectively equivalent to the call:</p>
	 * <code>Runtime.getRuntime().exit(n)</code>
	 *
	 * <p><strong>See important notes in Runtime.exit(int)</strong></p>
	 */
	public static void exit(int status) {
		//see important notes in Runtime.exit(int)
		Runtime.instance.exit(status);
	}

	/** IST-API
	 * Runs the garbage collector.
	 *
	 * Calling the gc method suggests that the Java Virtual Machine expend
	 * effort toward recycling unused objects in order to make the memory they
	 * currently occupy available for quick reuse. When control returns from
	 * the method call, the Java Virtual Machine has made a best effort to
	 * reclaim space from all discarded objects.
	 *
	 * The call System.gc() is effectively equivalent to the call:
	 *
	 * Runtime.getRuntime().gc()
	 */
	native public static void gc() ;

	/*
	 *  Gets the system property indicated by the specified key.
	 * Assumes that key is not null and key.length() > 0
	 */
	native public static String getPropertyNative(String key) ;

	/** IST-API
	 * Gets the system property indicated by the specified key.
	 */
	@Nullable 
	public static String getProperty(String key) {
		if (ej.bon.Constants.getBoolean(CalibrationConstants.CONSTANT_USE_SECURITYMANAGER)) {
			SecurityManager sm = getSecurityManager();
			if(sm != null){
				sm.checkPermission(new PropertyPermission(key, PropertyPermission.ACTION_READ));
			}
		}
		if (key.length==0) {
			throw new IllegalArgumentException() ;
		}
		return getPropertyNative(key) ;
	}

	@Nullable 
	public static String getPropertyNoCheck(String key){
		return getPropertyNative(key) ;
	}

	public static String getProperty(String key, String def) {
		String result = getProperty(key);
		if(result == null) {
			return def;
		}

		return result;
	}

	/** IST-API
	 * Returns the same hashcode for the given object as would be returned by
	 * the default method hashCode(), whether or not the given object's class
	 * overrides hashCode(). The hashcode for the null reference is zero.
	 */
	public static int identityHashCode(@Nullable Object x) {
		if (x==null) {
			return 0 ;
		}

		return hashCode(x) ;
	}

	private native static int hashCode(Object x);

	/*
	 * Non CLDC API methods, but intern implementation API.
	 * This class is fully VM dependent => add all methods that are fully VM dependent
	 */

	/*
	 * See specification in String.intern()
	 */
	public static native String intern(String source);

	private static OutputStream buildOutputStream()
	{
		try
		{
			String sysOutClass = getPropertyNoCheck(SYSTEM_OUT_CLASS_PROPERTY);	// assume the properties have been initialized
			assert sysOutClass != null;
			sysOutClass = sysOutClass.trim() ;
			Class<?> sysOut = Class.forName(sysOutClass) ;
			OutputStream outStream = (OutputStream) sysOut.newInstance() ;
			return outStream ;
		}
		catch (Throwable e) {
			// an error occurs
			// AssertionError, NullPointerException, ClassNotFoundException, ClassCastException, ...
			// This typically occurs when S3 boostraps the debug environment (special loader)
			return getOutputStream();
		}
	}

	/*
	 * Write the given stack trace in the given printstream.
	 * First element of stackTrace array is the number of stackFrame of the thread stack.
	 * Then this is a list of pairs of ips,methodAddr.
	 * If (stackTrace.length-1)/2 is lower than nbStackFrames, this means that stackTrace
	 * array does not contain all the stack frames. In this case, the number of remaining frames
	 * may be printed.
	 */
	public static void printStackTrace(PrintStream out, @Nullable int[] stackTrace, int indentation){
		String stackString = toStringStackTrace(stackTrace, indentation);
		if ( stackString == null) {
			out.print( "No stack description");
		}
		else {
			out.print(stackString);
		}
	}

	/**
	 * Returns the current value of the running Java Virtual Machine's high-resolution time source,
	 * in nanoseconds.
	 * 
	 * <p>
	 * This method can only be used to measure elapsed time and is not related to any other notion
	 * of system or wall-clock time. The value returned represents nanoseconds since some fixed but
	 * arbitrary <i>origin</i> time (perhaps in the future, so values may be negative). The same
	 * origin is used by all invocations of this method in an instance of a Java virtual machine;
	 * other virtual machine instances are likely to use a different origin.
	 * 
	 * <p>
	 * This method provides nanosecond precision, but not necessarily nanosecond resolution (that
	 * is, how frequently the value changes) - no guarantees are made except that the resolution is
	 * at least as good as that of {@link #currentTimeMillis()}.
	 * 
	 * <p>
	 * Differences in successive calls that span greater than approximately 292 years
	 * (2<sup>63</sup> nanoseconds) will not correctly compute elapsed time due to numerical
	 * overflow.
	 * 
	 * <p>
	 * The values returned by this method become meaningful only when the difference between two
	 * such values, obtained within the same instance of a Java virtual machine, is computed.
	 * 
	 * <p>
	 * For example, to measure how long some code takes to execute:
	 * 
	 * <pre>
	 * {
	 * 	&#064;code
	 * 	long startTime = System.nanoTime();
	 * 	// ... the code being measured ...
	 * 	long estimatedTime = System.nanoTime() - startTime;
	 * }
	 * </pre>
	 * 
	 * <p>
	 * To compare two nanoTime values
	 * 
	 * <pre>
	 * {@code
	 * long t0 = System.nanoTime();
	 * ...
	 * long t1 = System.nanoTime();}
	 * </pre>
	 * 
	 * one should use {@code t1 - t0 < 0}, not {@code t1 < t0}, because of the possibility of
	 * numerical overflow.
	 * 
	 * @return the current value of the running Java Virtual Machine's high-resolution time source,
	 *         in nanoseconds
	 */
	native public static long nanoTime();

	public static final native int[] stackIpsForOutOfMemoryError();

	/*
	 * Returns the String representation of the given stackTrace.
	 * stackTrace is a list of ips.
	 */
	@Nullable
	native public static String toStringStackTrace(@Nullable int[] stackTrace, int indentation);

	/* Returns the stack trace of a thread.
	 * This is a list of ips.
	 * 
	 * Returns null if the thread is terminated or not started
	 */
	@Nullable
	native public static int[] getStackTrace(Thread thread);

	/*default*/ static StackTraceElement[] getStackTraceElements(int[] ips){
		StackTraceElement[] stackTrace = new StackTraceElement[ips.length];

		int ipsLength = ips.length;
		for(int i=0 ; i<ipsLength ; ++i){
			stackTrace[i] = new StackTraceElement(ips[i]);
		}
		return stackTrace;
	}

	@Nullable 
	private static SecurityManager SecurityManager;
	public static void setSecurityManager(@Nullable SecurityManager s) {
		if (ej.bon.Constants.getBoolean(CalibrationConstants.CONSTANT_USE_SECURITYMANAGER)) {
			java.lang.SecurityManager currentSecurityManager = SecurityManager;
			if(currentSecurityManager != null){
				currentSecurityManager.checkPermission(new RuntimePermission("setSecurityManager"));
			}
		}
		SecurityManager = s;
	}

	@Nullable 
	public static SecurityManager getSecurityManager() {
		return SecurityManager;
	}
}
