/*
 * Java
 * 
 * Copyright 2008-2020 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.lang;


import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.util.Comparator;

import com.is2t.vm.support.lang.Systools;
import com.is2t.vm.support.util.EncodingConversion;

import ej.annotation.NonNull;
import ej.annotation.Nullable;

public final class String implements java.io.Serializable, Comparable<String>, CharSequence {
	
	private static class CaseInsensitiveOrder implements Comparator<String>, Serializable {
		// implements Serializable --> see Javadoc of CASE_INSENSITIVE_ORDER
		public int compare(String o1, String o2) {
			int len1 = o1.length();
			int len2 = o2.length();
			int minLen = Math.min(len1,  len2);
			
			for(int i = 0; i < minLen; i++) {
				char co1 = Character.toLowerCase(Character.toUpperCase(o1.charAt(i)));
				char co2 = Character.toLowerCase(Character.toUpperCase(o2.charAt(i)));
				if(co1 == co2) {
					continue;
				}
				
				// Characters are different here.
				return co1 - co2;
			}
			
			// One string is a substring of the other one.
			// Return length difference (so, 0 is string are equal)
			return len1 - len2;
		}
		
	}
	
	public static final Comparator<String> CASE_INSENSITIVE_ORDER = new CaseInsensitiveOrder();
	
	public static String copyValueOf(char[] data) {
		return copyValueOf(data, 0, data.length);
	}
	
	public static String copyValueOf(char[] data, int offset, int count) {
		return new String(data, offset, count);
	}

	/*
	 * Fields are well known by the underlying JVM
	 * - Do not change the order
	 * - Do not change types of the fields
	 */

	/*
	 * char array (maybe shared between several String => String literals for example)
	 * the length of this char array should not be used to get the length of the string
	 */
	public char[] chars ; // = null VM known

	/*
	 * Index of the first character of the string
	 */
	public int offset ; // == 0 VM known

	/*
	 * Number of characters in the string (see chars)
	 */
	public int length ; // == 0 VM known

	/*
	 * Intern hash code cache
	 */
	private int hashcode ;

	public String() {
		// no need to reinitialize the fields length and offset
		this.chars = new char[0] ;
	}

	public String(byte[] bytes) {
		this(bytes, 0, bytes.length) ;
	}

	public String(byte[] bytes, int off, int len){
		this(bytes, off, len, EncodingConversion.DefaultEncoding);
	}

	public String(byte[] bytes, int off, int len, String enc) throws UnsupportedEncodingException {
		this(bytes, off, len, EncodingConversion.getEncoding(enc));
	}
	
	private String(byte[] bytes, int offset, int length, EncodingConversion encoding){
		char[] chars = new char[length];
		int nbChars = encoding.decode(bytes, new int[]{offset}, length, chars, 0, chars.length);
		if(nbChars != chars.length){
			// resize
			char[] newChars = new char[nbChars];
			System.arraycopy(chars, 0, newChars, 0, nbChars);
			chars = newChars;
		}
		this.chars = chars;
		//this.offset = 0; default value
		this.length = chars.length;
	}

	public String(byte[] bytes, String enc) throws UnsupportedEncodingException {
		this(bytes, 0, bytes.length, enc) ;
	}

	public String(char[] value) {
		this(value, 0, value.length) ;
	}

	public String(char[] value, int offset, int count) {
		this(value, offset, count, true) ;
	}

	public String(String value) {
		this(value.chars, value.offset, value.length, true) ;
	}

	public String(StringBuffer buffer) {
		// All methods in StringBuffer are synchronized
		this.length = buffer.length() ;
		buffer.getChars(0, length, this.chars = new char[this.length], 0) ;
	}
	
	public String(StringBuilder builder) {
		this.length = builder.length() ;
		builder.getChars(0, length, this.chars = new char[this.length], 0) ;
	}

	public char charAt(int index) {
		// Need to test the index validity because the array may be larger than the String wrapper
		if(index < 0 || index >= length) {
			throw new IndexOutOfBoundsException();
		}
		return this.chars[this.offset+index] ;
	}

	/**
	 * @accelerable
	 */
	public int compareTo(String anotherString) {
		// iteration on min of both lengths
		char[] array = this.chars;
		char[] otherArray = anotherString.chars; // will throw a NPE as specified by java.lang.Comparable<java.lang.String>.compareTo
		int length = this.length;
		int anotherStringLength = anotherString.length;
		int minLength = length < anotherStringLength ? length : anotherStringLength;
		int stop = offset + minLength;
		int i = this.offset-1;
		int j = anotherString.offset-1;

		while (++i<stop) {
			if(array[i] != otherArray[++j]) {
				return array[i] - otherArray[j];
			}
		}
		return this.length - anotherString.length;
	}
	
	public int compareToIgnoreCase(String str) {
		return CASE_INSENSITIVE_ORDER.compare(this, str);
	}

	public String concat(String str) {
		// Throws NullPointerException if str==null
		// Returns this if str.length()==0

		int strLen = str.length ;
		int llength = this.length ;

		if (strLen > 0) {
			int newLength = llength+strLen ; // may overflow
			if (newLength<0) {
				// String lengths are stored in positive integers. So if someone
				// wants to concatenate a string which makes the result's length
				// greater than MAX_INTEGER, we must cancel this operation.
				newLength = Integer.MAX_VALUE ;
				strLen = newLength-llength ;
			}
			char[] newArray = new char[newLength] ;
			System.arraycopy(this.chars, this.offset, newArray, 0, llength) ;
			System.arraycopy(str.chars, str.offset, newArray, llength, strLen) ;
			return new String(newArray, 0, newLength, false) ;
		}
		return this ;
	}
	
	public boolean contains(CharSequence s) {
		//FIXME See if this.lastIndexOf is more efficient
		return this.indexOf(s.toString()) > -1;
	}
	
	public boolean contentEquals(CharSequence cs) {
		return this.compareTo(cs.toString()) == 0;
	}
	
	public boolean contentEquals(StringBuffer sb) {
		synchronized (sb) {
			return this.contentEquals(sb.toString());
		}
	}

	public boolean endsWith(String suffix) {
		return suffix.length==0 || Systools.regionMatches(this,this.length-suffix.length, suffix, 0, suffix.length) ;
	}

	@Override
	public native boolean equals(@Nullable Object anObject);

	/**
	 * @accelerable
	 */
	public boolean equalsIgnoreCase(@Nullable String other){
		if(other == null){
			return false; // not a string or other is null
		}
		
		if(this == other){
			// same reference
			return true;
		}
		
		int length = this.length;
		if(length != other.length) {
			return false;
		}
		
		char[] array = this.chars;
		int offset = this.offset;
		char[] otherArray = other.chars;
		int otherOffset = other.offset;
		
		for(int i=-1; ++i<length;){
			char c1 = array[offset+i];
			char c2 = otherArray[otherOffset+i];
			// see CDLC-1.1 spec
			// Two characters c1 and c2 are considered the same, ignoring case if at least one of the following is true:
			if(!(
					// The two characters are the same (as compared by the == operator).
					c1 == c2 ||
					// Applying the method Character.toUpperCase(char) to each character produces the same result.
					Character.toUpperCase(c1) == Character.toUpperCase(c2) ||
					// Applying the method Character.toLowerCase(char) to each character produces the same result.
					Character.toLowerCase(c1) == Character.toLowerCase(c2)
			)) {
				return false;
			}
		}
		
		return true;
	}

	public byte[] getBytes() {
		return getBytes(EncodingConversion.DefaultEncoding);
	}

	private byte[] getBytes(EncodingConversion encoding) {
		byte[] bytes = new byte[length*encoding.getMaxBytesPerChar()];
		int nbBytes = encoding.encode(chars,new int[]{offset}, length, bytes, 0, bytes.length);
		if(length != bytes.length){
			// resize
			byte[] newBytes = new byte[nbBytes];
			System.arraycopy(bytes, 0, newBytes, 0, nbBytes);
			return newBytes;
		}
		
		return bytes;
	}

	public byte[] getBytes(String charsetName) throws UnsupportedEncodingException {
		return getBytes(EncodingConversion.getEncoding(charsetName));
	}

	/**
	 * Transforms this Java String into a C String.<br>
	 * The platform default encoding is used to transform Java characters into C characters.<br>
	 * The created C String is a NULL terminated String (ends with '\0'). The <code>cString</code> array length
	 * must be at least <code>this.length()+1</code>.
	 * 
	 * @param cString byte array which contains the C String.
	 * 
	 * @throws ArrayIndexOutOfBoundsException if cString is too small to contain the string.
	 */
	public void toCString(byte[] cString) {
		// check string length
		if (cString.length < length + 1) {
			throw new ArrayIndexOutOfBoundsException() ;
		}
		
		EncodingConversion encoding = EncodingConversion.DefaultEncoding;
		int nbBytes = encoding.encode(chars, new int[]{offset}, length, cString, 0, length);		
		cString[nbBytes] = 0 ;	// C String is NULL terminated
	}

	public void getChars(int srcBegin, int srcEnd, char[] dst, int dstBegin) {
        if(srcBegin<0 || srcEnd>length) {// must be checked because it is not correctly checked in the array copy because the String is a subset of the array
            throw new IndexOutOfBoundsException();
        }
        System.arraycopy(this.chars, this.offset+srcBegin, dst, dstBegin, srcEnd-srcBegin);
	}

	@Override
	public int hashCode() {
		//WARNING /!\ implementation duplicated in VMStructAccess.iceTea for hashtable optimization
		
		if (hashcode == 0 && this.length!=0){//don't compute hashcode for empty strings or if it is already computed
			int hashcodeTmp = hashCodeNative() ;
			if(hashcodeTmp != 0){
				//computed hashcode may be 0 if the string contains '\0' characters. If this string is literal (i.e. in flash)
				//we must not write in it (putfield not allowed on immutable objects)
				this.hashcode = hashcodeTmp;
			}
		}
		return hashcode ;
	}

	/*
	 * This method should be implemented in native side (VM specific)
	 */
	public native int hashCodeNative() ;

	public int indexOf(int ch) {
		return indexOf(ch, 0) ;
	}

	/*
	 * This method should be implemented in native side for speed considerations
	 */
	public native int indexOf(int ch, int fromIndex) ;

	public int indexOf(String str) {
		return indexOf(str, 0) ;
	}

	// Cyclomatic complexity : acceptable
	public int indexOf(String str, int fromIndex) { //NOSONAR
		if(fromIndex < 0) {
			fromIndex = 0 ;
		}

		// Avoid getfield opcode
		int strLen = str.length ;
		int thisLen = this.length ;

		if(fromIndex >= thisLen || thisLen-fromIndex < strLen) {
			if(fromIndex==thisLen && thisLen==0 && strLen==0) {
				return 0;//empty string may be find at index 0 of an empty string
			}
			return -1;
		}

		if(strLen==0) {
			return fromIndex;
		}

		// Avoid getfield opcode
		int off = this.offset ;
		int strOff = str.offset ;
		char[] chars = this.chars ;
		char[] strChars = str.chars ;

		int end = off + thisLen - strLen ;
		off += fromIndex;
		if(off < 0) {
			off = this.offset ;
		}

		char firstChar = strChars[strOff];

		beginSearchOfFirstChar:
			while (true) {
				while (off <= end && chars[off] != firstChar) {
					off++ ;
				}

				if (off <= end && chars[off]==firstChar) {
					for (int i=strLen; --i>=0; ){
						if (strChars[strOff+i] != chars[off+i]) {
							off++ ;
							continue beginSearchOfFirstChar ;
						}
					}

					return off-this.offset ;
				}
				
				return -1 ;
			}
	}

	public String intern(){
		return System.intern(this);
	}
	
	public boolean isEmpty() {
		return this.length == 0;
	}

	public int lastIndexOf(int ch) {
		return lastIndexOf(ch, length-1) ;
	}

	/*
	 * This method should be implemented in native side for speed considerations
	 */
	public native int lastIndexOf(int ch, int fromIndex) ;

	public int length() {
		return this.length ;
	}

	public boolean regionMatches(boolean ignoreCase, int toffset, @Nullable String other, int ooffset, int len) {
		if (!ignoreCase){
			if(other == null) {
				throw new NullPointerException(); // NOSONAR
			}
			return Systools.regionMatches(this, toffset, other, ooffset, len);
		}
		else {
			String s1 = toLowerCase();
			String s2 = other.toLowerCase();
			return Systools.regionMatches(s1,toffset, s2, ooffset, len);
		}
	}
	
	public boolean regionMatches(int toffset, String other, int ooffset, int len) {
		return this.regionMatches(false, toffset, other, ooffset, len);
	}

	public String replace(char oldChar, char newChar) {

		if(this.indexOf(oldChar)==-1) {
			return this;
		}

		int length = this.length ;
		int offset = this.offset ;

		char[] newArray = new char[length];
		char[] chars = this.chars;

		for (int i=offset+length; --i>=offset; ) {
			newArray[i-offset] = ((chars[i] == oldChar) ? newChar : chars[i]);
		}

		// Constructs a new String object with a non-shared array
		// See the constructor "String(char[],int,int,boolean)" for more details.
		return new String(newArray, 0, length, false);
	}
	
	public String replace(CharSequence target, CharSequence replacement) {
		int startIndex = 0;
		String targetAsString = target.toString();
		int targetLen = target.length();
		int index = this.indexOf(targetAsString);
		
		if(index == -1) {
			// Nothing to do!
			// We can not avoid String instantiation in String class
			return new String(this); //NOSONAR
		}
		
		// At least one match
		StringBuilder b = new StringBuilder();
		while(index != -1) {
			b.append(this, startIndex, index).append(replacement);
			startIndex = index + targetLen;
			
			index = this.indexOf(targetAsString, startIndex);
		}
		b.append(this, startIndex, this.length);
		
		return b.toString();
	}

	public boolean startsWith(String prefix) {
		return regionMatches(false, 0, prefix, 0, prefix.length) ;
	}

	public boolean startsWith(String prefix, int toffset) {
		return regionMatches(false, toffset, prefix, 0, prefix.length) ;
	}

	public String substring(int beginIndex) {
		if(beginIndex < 0 || beginIndex > this.length) {//don't made by the new String because the character string is a subset of this.Chars()
			throw new IndexOutOfBoundsException() ;
		}
		return new String(this.chars, this.offset+beginIndex, this.length-beginIndex, false);
	}

	public String substring(int beginIndex, int endIndex) {
		//  endIndex < beginIndex  are already tested  in new String() :
		if(beginIndex < 0 || beginIndex > this.length ||endIndex>length) {
			throw new IndexOutOfBoundsException();		//don't made by the new String because the character string is a subset of this.Chars()
		}
		return new String(this.chars, this.offset+beginIndex, endIndex-beginIndex, false);
	}

	public char[] toCharArray() {
		char[] newArray ;
		System.arraycopy(this.chars, this.offset,
				newArray = new char[this.length],
				0, this.length) ;
		return newArray ;
	}

	public String toLowerCase() {
		char[] chars = this.chars;
		char[] newChars = new char[length];
		int offset = this.offset;
		for(int i=length; --i>=0;){
			newChars[i] = Character.toLowerCase(chars[i+offset]);
		}
		return new String(newChars, 0, length, false);
	}

	@Override
	public String toString() {
		return this ;
	}

	public String toUpperCase() {
		char[] chars = this.chars;
		char[] newChars = new char[length];
		int offset = this.offset;
		for(int i=length; --i>=0;){
			newChars[i] = Character.toUpperCase(chars[i+offset]);
		}
		return new String(newChars, 0, length, false);
	}

	public String trim() {

		if (this.length > 0) { // else, nothing to do!

			int offset;
			int length = this.length;
			int i = offset=this.offset ;   // first index of char less than ' '
			int j = offset + length - 1;  // last index of char less than ' '
			char[] array = this.chars ;

			// Find the first character whose value is less than ' '
			while(array[i] <= ' ' && i<j) {
				++i ;
			}

			// Find the last character whose value is less than ' '
			while (j>=i && array[j] <= ' ') {
				--j ;
			}

			//if there is something to trim
			if(i!=offset || j!= (offset+length - 1)) {
				if (i==j && array[i] <= ' ') {// The string is empty
					// Sonar : avoid String instantiation, but we are in String class
					return new String(); // NOSONAR
				}
					
				return new String(array, i, j-i+1, false);
			}
		}
		return this ;
	}

	public static String valueOf(boolean b) {
		return Boolean.toString(b);
	}

	public static String valueOf(char c) {
		return new String(new char[]{c}, 0, 1, false) ;
	}

	public static String valueOf(char[] data) {
		return new String(data, 0, data.length) ;
	}

	public static String valueOf(char[] data, int offset, int count) {
		return new String(data, offset, count) ;
	}

	public static String valueOf(double d) {
		return Double.toString(d) ;
	}

	public static String valueOf(float f) {
		return Float.toString(f) ;
	}

	public static String valueOf(int i) {
		return Integer.toString(i, 10) ;
	}

	public static String valueOf(long l) {
		return Long.toString(l) ;
	}

	public static String valueOf(@Nullable Object obj) {
		@SuppressWarnings("null")
		@NonNull String result = obj != null ? obj.toString() : "null" ;//$NON-NLS-1$
		return result;
	}

	/*
	 * NON CLDC APIs methods and constructors - Allowed to be declared public because they
	 * cannot be overridden by user (String is final)
	 */

	/*
	 * Creates a new String object without allocating a new
	 * char array. The reference to 'src' is just copied.
	 * Such a kind of constructor is useful to avoid the copy of
	 * each element contained in 'chars' when you create a new
	 * immutable String or when you have to create a new String with
	 * a shared array which must be "deep copied" (ex: String.valueOf(char[])).
	 */
	public String(char[] src, int srcOffset, int srcLength, boolean needCopy){ // NOSONAR
		// Sonar : Array "src" is stored directly
		// the boolean "needCopy" indicates if src must be copy are not.
		if(srcOffset <0 || srcOffset>src.length || srcLength<0 || src.length-srcLength<srcOffset) {
			throw new IndexOutOfBoundsException() ;
		}

		if(!needCopy) {
			this.chars = src ;
			this.offset = srcOffset ;
			this.length = srcLength ;
		}
		else { // we must copy the contents of the char array
			System.arraycopy(src, srcOffset, this.chars = new char[srcLength], this.offset=0, this.length=srcLength) ;
		}
	}
	
	public int lastIndexOf(String str) {
		return this.lastIndexOf(str, this.length);
	}
	
	
	// Cyclomatic Complexity : Cyclomatic Complexity is 15 (max allowed is 10).
	// acceptable here
	public int lastIndexOf(String str, int fromIndex) { //NOSONAR
		int targetLength = str.length;
		int sourceLength = this.length;
		if (fromIndex < 0) {
			return -1;
		}
		
		int rightIndex = sourceLength - targetLength;
		if(fromIndex > rightIndex){
			fromIndex = rightIndex;
		}
		
		if(targetLength == 0) {
			return fromIndex; //Empty string may be find at fromIndex of this string. Always matches
		}

		int end = targetLength - 1;
		int posInThis = fromIndex + end;
		char lastChar = str.charAt(end);

		beginSearchOfFirstChar:
			while(true) {
				while(posInThis >= end && this.charAt(posInThis) != lastChar) {
					posInThis--;
				}
				if(posInThis < end) {
					return -1;
				}
				
				// The last char matches. Check other chars
				int j = posInThis;
				for(int i = 0; i < targetLength - 1; i++) {
					if(this.charAt(j - i) != str.charAt(str.length - 1 - i)) {
						posInThis--;
						continue beginSearchOfFirstChar;
					}
				}
				
				return j - targetLength + 1;
			}
	}

	
	public CharSequence subSequence(int start, int end) {
		return this.substring(start, end);
	}
	
}