/*
 * Java
 * 
 * Copyright 2013-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.lang;

import ej.annotation.Nullable;

/*
 * Implement StackTraceElement
 * 
 */
public final class StackTraceElement implements java.io.Serializable {


	/**
	 * Internal pointer to the bytecode
	 */
	private int ip;
	
	
	StackTraceElement() {
		throw new RuntimeException();
	}
	
	StackTraceElement(int ip) {
		this.ip = ip;
	}

	/**
	 * Returns true if the specified object is another {@code StackTraceElement} instance
	 * representing the same execution point as this instance. Two stack trace elements {@code a}
	 * and {@code b} are equal if and only if:
	 * 
	 * <pre>
	 *     equals(a.getFileName(), b.getFileName()) &&
	 *     a.getLineNumber() == b.getLineNumber()) &&
	 *     equals(a.getClassName(), b.getClassName()) &&
	 *     equals(a.getMethodName(), b.getMethodName())
	 * </pre>
	 * 
	 * @param obj
	 *        the object to be compared with this stack trace element.
	 * @return true if the specified object is another {@code StackTraceElement} instance
	 *         representing the same execution point as this instance.
	 */
	@Override
	public boolean equals(@Nullable Object obj) {
		if(obj == this){
			return true;
		}
		
		if(!(obj instanceof StackTraceElement)){
			return false;
		}
		
		StackTraceElement other = (StackTraceElement) obj;

		return equals(getFileName(), other.getFileName()) &&
			   (getLineNumber() == other.getLineNumber()) &&
			   equals(getClassName(), other.getClassName()) &&
			   equals(getMethodName(), other.getMethodName());
	}
	

	/**
	 * Returns a hash code value for this stack trace element.
	 */
	@Override
	public int hashCode() {
		return hashCode(getFileName())+getLineNumber()+hashCode(getClassName())+hashCode(getMethodName());
	}
	
	
	private static boolean equals(@Nullable Object o1, @Nullable Object o2){
		if(o1 == null){
			return o2 == null;
		}
		else {
			return o1.equals(o2);
		}
	}
	
	private static int hashCode(@Nullable Object o){
		if(o == null){
			return 0;
		}
		else {
			return o.hashCode();
		}
	}

	/**
	 * Returns the fully qualified name of the class containing the execution point represented by
	 * this stack trace element.
	 * 
	 * @return the fully qualified name of the {@code Class} containing the execution point
	 *         represented by this stack trace element.
	 */
	public String getClassName() {
		Class<?> clazz = getClass(ip);
		if(clazz != null){
			return clazz.getName();
		}
		else {
			return "";
		}
	}

	/**
	 * Returns the name of the source file containing the execution point represented by this stack
	 * trace element. Generally, this corresponds to the {@code SourceFile} attribute of the
	 * relevant {@code class} file (as per <i>The Java Virtual Machine Specification</i>, Section
	 * 4.7.7). In some systems, the name may refer to some source code unit other than a file, such
	 * as an entry in source repository.
	 * 
	 * @return the name of the file containing the execution point represented by this stack trace
	 *         element, or {@code null} if this information is unavailable.
	 */
	@Nullable 
	public String getFileName() {
		return getFileName(ip);
	}

	/**
	 * Returns the line number of the source line containing the execution point represented by this
	 * stack trace element. Generally, this is derived from the {@code LineNumberTable} attribute of
	 * the relevant {@code class} file (as per <i>The Java Virtual Machine Specification</i>,
	 * Section 4.7.8).
	 * 
	 * @return the line number of the source line containing the execution point represented by this
	 *         stack trace element, or a negative number if this information is unavailable.
	 */
	public int getLineNumber() {
		return getLineNumber(ip);
	}

	/**
	 * Returns the name of the method containing the execution point represented by this stack trace
	 * element. If the execution point is contained in an instance or class initializer, this method
	 * will return the appropriate <i>special method name</i>, {@code <init>} or {@code <clinit>},
	 * as per Section 3.9 of <i>The Java Virtual Machine Specification</i>.
	 * 
	 * @return the name of the method containing the execution point represented by this stack trace
	 *         element.
	 */
	public String getMethodName() {
		return getMethodName(ip);
	}


	/**
	 * Returns a string representation of this stack trace element. The format of this string
	 * depends on the implementation.
	 */
	@Override
	public String toString() {
		return getClassName()+"."+getMethodName()+"("+getFileName()+": "+getLineNumber()+")";
	}
	
	@Nullable
	native static public Class<?> getClass(int ip);

	native static public String getFileName(int ip);

	native static public int getLineNumber(int ip);

	native static public String getMethodName(int ip);

}
