/*
 * Java
 *
 * Copyright 2008-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.lang;

import com.is2t.vm.support.CalibrationConstants;

/** IST-API
 * Every Java application has a single instance of class Runtime that allows
 * the application to interface with the environment in which the application
 * is running. The current runtime can be obtained from the getRuntime method.
 * 
 * An application cannot create its own instance of this class.
 */
public class Runtime extends Object {
	
	// singleton obj
	static final Runtime instance = new Runtime() ;
	
	/** IST-API
	 * <p>Terminates the currently running Java application. This method never
	 * returns normally.</p>
	 * <p>
	 * IST specific feature:
	 * IST implementation returns normally and executes finally blocks of
	 * all running threads. This is safer if developpers have important
	 * stuff within their finally blocks, such as deallocating file descriptors
	 * or unbinding sockets.</p>
	 * <p>
	 * <strong>Right now, note that IST virtual machine is not responsible for
	 * closing pending streams when Runtime.exit() method is called.</strong></p>
     * 
	 * The argument serves as a status code; by convention, a nonzero status
	 * code indicates abnormal termination.
	 */
	public void exit(int status){
		if (ej.bon.Constants.getBoolean(CalibrationConstants.CONSTANT_USE_SECURITYMANAGER)) {
			SecurityManager securityManager = System.getSecurityManager();
			if (securityManager != null) {
				securityManager.checkPermission(new RuntimePermission("exitVM."+status));
			}
		}
		exit0(status);
	}
	
	native public void exit0(int status)
		//IMPORTANT NOTE:
		//IST implementation returns normally and executes finally blocks of
		//all running threads. This is safer if developpers have important
		//stuff within their finally blocks, such as deallocating file descriptors
		//or unbinding sockets.
		//
		//Right now, note that IST virtual machine is not responsible for
		//closing pending streams when the Runtime.exit() method is called.
	;
	
	/** IST-API
	 * Returns the amount of free memory in the system. Calling the gc method
	 * may result in increasing the value returned by freeMemory.
	 */
	native public long freeMemory() ;
	
	/** IST-API
	 * Runs the garbage collector. Calling this method suggests that the Java
	 * Virtual Machine expend effort toward recycling unused objects in order
	 * to make the memory they currently occupy available for quick reuse. When
	 * control returns from the method call, the Java Virtual Machine has made
	 * its best effort to recycle all discarded objects.
	 * 
	 * The name gc stands for "garbage collector". The Java Virtual Machine
	 * performs this recycling process automatically as needed even if the gc
	 * method is not invoked explicitly.
	 * 
	 * The method System.gc() is the conventional and convenient means of
	 * invoking this method.
	 */
	native public void gc() ;
	
	/** IST-API
	 * Returns the runtime object associated with the current Java application.
	 * Most of the methods of class Runtime are instance methods and must be
	 * invoked with respect to the current runtime object.
	 */
	public static Runtime getRuntime() {
		return Runtime.instance ;
	}
	
	/**
	 * Returns the maximum amount of memory that the Java virtual machine will attempt to use. If
	 * there is no inherent limit then the value {@link java.lang.Long#MAX_VALUE} will be returned.
	 * </p>
	 * 
	 * @return the maximum amount of memory that the virtual machine will attempt to use, measured
	 *         in bytes
	 */
	public long maxMemory() {
		return totalMemory();
	}
	 
	/** IST-API
	 * Returns the total amount of memory in the Java Virtual Machine. The
	 * value returned by this method may vary over time, depending on the host
	 * environment.
	 * 
	 * Note that the amount of memory required to hold an object of any given
	 * type may be implementation-dependent.
	 */
	native public long totalMemory() ;
	
}
