/*
 * Java
 *
 * Copyright 2008-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.lang;

import java.util.Random;

import ej.annotation.Nullable;

public final class Math extends Object {

	public static final double E = 2.7182818284590452354;
	public static final double PI = 3.14159265358979323846;

	@Nullable
	private static Random RandomGenerator;

    private Math(){ /* Math constructor cannot be instantiated */ }

	public static double abs(double a) {
		//basically, set the highest bit to zero
		if (a > 0) {
			return a;		//	  optimization when a > 0 or a=NaN
		}

		return Double.longBitsToDouble((Double.doubleToLongBits(a)<<1)>>>1) ;
	}

	public static float abs(float a) {
		//basically, set the highest bit to 0.
		if (a > 0) {
			return a;		//	  optimization when a > 0 or a=NaN
		}

		return Float.intBitsToFloat(0x7fffffff & Float.floatToIntBits(a)) ;
	}

	/*
	 * This native method may be SOAR-known and turned into an opcode
	 */
	public static native int abs(int a);

	/**
	 * @accelerable
	 */
	public static long abs(long a){
		return (a < 0 && a > Long.MIN_VALUE) ? -a : a;
	}

	public static native double acos(double a);

	public static native double asin(double a);

	public static native double atan(double a);

	public static native double atan2(double y, double x);

	public static native double cbrt(double a);

	public static native double ceil(double a);

	public static double copySign(double magnitude, double sign){
		return copySignD(magnitude, sign);
	}

	public static native double copySignD(double magnitude, double sign) ;

	public static float copySign(float magnitude, float sign){
		return copySignF(magnitude, sign);
	}

	public static native float copySignF(float magnitude, float sign) ;

	public static native double cos(double a);

	public static native double cosh(double x);

	public static native double exp(double a);

	public static native double expm1(double x);

	public static native double floor(double a) ;

	public static int getExponent(double d){
		long l = Double.doubleToRawLongBits(d);
		long exp = (l >>> 52) & 0x7FF;
		return (int)(exp - Double.MAX_EXPONENT);
	}

	public static int getExponent(float f){
		int i = Float.floatToRawIntBits(f);
		int exp = (i >>> 23) & 0xFF;
		return exp - Float.MAX_EXPONENT;
	}

	public static native double hypot(double x, double y);

	// Method Name : Name 'IEEEremainder' must match pattern '^[a-z][a-zA-Z0-9_]*$'.
	// cannot do anything!
	public static native double IEEEremainder(double f1, double f2); //NOSONAR

	public static native double log(double a);

	public static native double log10(double a);

	public static native double log1p(double x);

	/**
	 * @accelerable
	 */
	public static double max(double a, double b) {
		if(a > b) {
			return a;
		}
		if(b > a) {
			return b;
		}
		//here a==b but when a or be is NaN
		if(a != b) {
			return Double.NaN; // see 4.2.3 of the Java Language Specification
		}
		//there is TWO zeros in java +0.0 and -0.0 and -0.0 is the smallest :-(
		if(a == 0.0) {
			//if either a or b is -0.0, then +0.0 !!! (note +0.0==-0.0)
			if(Double.doubleToLongBits(a)!=0L) {
				return b;
			}
			if(Double.doubleToLongBits(b)!=0L) {
				return a;
			}
		}
		return a;
	}

	/**
	 * @accelerable
	 */
	public static float max(float a, float b) {
		if(a > b) {
			return a;
		}
		if(b > a) {
			return b;
		}
		//here a==b but when a or be is NaN
		if(a != b) {
			return Float.NaN; // sec.4.2.3 of the Java Language Specification
		}
		//there is TWO zeros in java +0.0f and -0.0f and -0.0f is the smallest :-(
		if(a == 0.0f) {
			//if either a or b is -0.0f, then return +0.0 !!! (note +0.0f==-0.0f)
			if(Float.floatToIntBits(a)!=0) {
				return b;
			}
			if(Float.floatToIntBits(b)!=0) {
				return a;
			}
		}
		return a;
	}

	/*
	 * This native method may be SOAR-known and turned into an opcode
	 */
	public static native int max(int a, int b);

	/**
	 * @accelerable
	 */
	public static long max(long a, long b) {
		return a > b ? a : b;
	}

	/**
	 * @accelerable
	 */
	public static double min(double a, double b) {
		if(b > a) {
			return a;
		}
		if(a > b) {
			return b;
		}
		//here a==b but when a or be is NaN
		if(a != b) {
			return Double.NaN; // sec.4.2.3 of the Java Language Specification
		}
		//there is TWO zero in java +0.0 and -0.0 and -0.0 is the smallest :-(
		if(a == 0.0) {
			//if either a or b is -0.0, then return it !!! (note +0.0==-0.0)
			if(Double.doubleToLongBits(a)!=0L) {
				return a;
			}
			if(Double.doubleToLongBits(b)!=0L) {
				return b;
			}
		}
		return a;
	}

	/**
	 * @accelerable
	 */
	public static float min(float a, float b) {
		if(b > a) {
			return a;
		}
		if(a > b) {
			return b;
		}
		//here a==b but when a or be is NaN
		if(a != b) {
			return Float.NaN; // sec.4.2.3 of the Java Language Specification
		}
		//there is TWO zeros in java +0.0f and -0.0f and -0.0f is the smallest :-(
		if(a == 0.0f) {
			//if either a or b is -0.0f, then return it !!! (note +0.0f==-0.0f)
			if(Float.floatToIntBits(a)!=0) {
				return a;
			}
			if(Float.floatToIntBits(b)!=0) {
				return b;
			}
		}
		return a;
	}

	/*
	 * This native method may be SOAR-known and turned into an opcode
	 */
	public static native int min(int a, int b);

	public static long min(long a, long b){
		return a < b ? a : b;
	}

	public static double nextAfter(double start, double direction){
		return nextAfterD(start, direction);
	}

	public static native double nextAfterD(double start, double direction);

	public static float nextAfter(float start, double direction){
		return nextAfterF(start, direction);
	}

	public static native float nextAfterF(float start, double direction);

	public static double nextUp(double d){
		return nextAfterD(d, Double.POSITIVE_INFINITY);
	}

	public static float nextUp(float f){
		return nextAfterF(f, Float.POSITIVE_INFINITY);
	}

	public static native double pow(double a, double b);

	public static double random() {
		Random randomGenerator = RandomGenerator;
		if(randomGenerator == null) {
			RandomGenerator = randomGenerator = new Random();
		}
		return randomGenerator.nextDouble();
	}

	public static native double rint(double a);

	/**
	 * @accelerable
	 */
	public static long round(double a) {
		if(Double.isNaN(a)) {
			return 0;
		}

		if(a == Double.NEGATIVE_INFINITY || a <= Long.MIN_VALUE) {
			return Long.MIN_VALUE;
		}

		if(a == Double.POSITIVE_INFINITY || a >= Long.MAX_VALUE) {
			return Long.MAX_VALUE;
		}

		double hop = (a > 0 ? 0.5d : -0.5d);
		double dres = a + hop;
		long res = (long) dres;
		if(res < 0 && res == dres){
			return res + 1;
		}
		else {
			return res;
		}
	}

	/**
	 * @accelerable
	 */
	public static int round(float a) {
		if(Float.isNaN(a)) {
			return 0;
		}

		if(a == Float.NEGATIVE_INFINITY || a <= Integer.MIN_VALUE) {
			return Integer.MIN_VALUE;
		}

		if(a == Float.POSITIVE_INFINITY || a >= Integer.MAX_VALUE) {
			return Integer.MAX_VALUE;
		}

		double hop = (a > 0 ? 0.5f : -0.5f);
		return (int) (a + hop);
	}

	public static double scalb(double d, int scaleFactor){
		return scalbD(d, scaleFactor);
	}

	public static native double scalbD(double d, int scaleFactor);

	public static float scalb(float f, int scaleFactor){
		return scalbF(f, scaleFactor);
	}

	public static native float scalbF(float f, int scaleFactor);

	/**
	 * @accelerable
	 */
	public static double signum(double d) {
		if(Double.isNaN(d)) {
			return Double.NaN;
		}

		if(d == 0d) {
			return 0;
		}
		else if(d < 0d) {
			return -1;
		}
		else {
			return 1;
		}
	}

	/**
	 * @accelerable
	 */
	public static float signum(float f) {
		if(Float.isNaN(f)) {
			return Float.NaN;
		}

		if(f == 0f) {
			return 0;
		}
		else if(f < 0f) {
			return -1;
		}
		else {
			return 1;
		}
	}

	public static native double sin(double a) ;

	public static native double sinh(double x);

	public static native double sqrt(double a) ;

	public static native double tan(double a) ;

	public static native double tanh(double x);

	/**
	 * @accelerable
	 */
	public static double toDegrees(double angrad) {
		return angrad / PI * 180 ;
	}

	/**
	 * @accelerable
	 */
	public static double toRadians(double angdeg) {
		return angdeg / 180 * PI ;
	}

	public static double ulp(double d){
		return nextUp(d) - d;
	}

	public static float ulp(float f){
		return nextUp(f) - f;
	}

}
