/*
 * Java
 * 
 * Copyright 2008-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.lang;

import com.is2t.vm.support.lang.SupportNumber;

import ej.annotation.Nullable;

public final class Integer extends Number implements Comparable<Integer> {

	public static final int MAX_VALUE = 0x7fffffff;

	public static final int MIN_VALUE = 0x80000000;

	public static final int SIZE = 32;
	
//	public static int bitCount(int i) {
//	}
	
	/**
	 * @accelerable
	 */
	public static int compare(int x, int y) {
		if(x == y) {
			return 0;
		}
		else if(x < y) {
			return -1;
		}
		else {
			return 1;
		}
	}
	
	public static Integer decode(String nm) throws NumberFormatException {
		// Steps: 
		// - decode as a long
		// - check bounds
		
		// step 1
		long decode = Long.decode(nm).longValue();
		
		// step 2
		if(decode > MAX_VALUE || decode < MIN_VALUE) {
			throw new NumberFormatException();
		}
		
		return valueOf((int) decode); // cast is safe
	}
	
	@Nullable 
	public static Integer getInteger(String nm) {
		return getInteger(nm, null);
	}
	
	public static Integer getInteger(String nm, int val) {
		Integer result = getInteger(nm, valueOf(val));
		assert result != null;
		return result;
	}
	
	public static Integer getInteger(String nm, Integer val) {
		// Steps: 
		// - get as a long
		// - check bounds
		
		//step 1
		Long value = Long.getLong(nm);
		long longValue = (value == null ? 0 : value.longValue());
		
		// step 2
		if(value == null || longValue > MAX_VALUE || longValue < MIN_VALUE) {
			return val;
		}
		
		return valueOf((int) longValue); // cast is safe
	}
	
//	public static int highestOneBit(int i) {
//	}
//	
//	public static int lowestOneBit(int i) {
//	}
//	
//	public static int numberOfLeadingZeros(int i) {
//	}
//	
//	public static int numberOfTrailingZeros(int i) {
//	}

	public static int parseInt(@Nullable String s) throws NumberFormatException {
		return Integer.parseInt(s, 10);
	}

	public static int parseInt(@Nullable String s, int radix) throws NumberFormatException {
		// Steps: 
		// - parse as a long
		// - check bounds
		
		// step 1
		long parse = Long.parseLong(s, radix);
		
		// step 2
		if(parse > MAX_VALUE || parse < MIN_VALUE) {
			throw new NumberFormatException();
		}
		
		return (int) parse; // cast is safe
	}
	
//	public static int reverse(int i) {
//	}
	
//	public static int reverseBytes(int i) {
//	}
	
	/**
	 * @accelerable
	 */
	public static int rotateLeft(int i, int distance) {
		return (i << distance) | (i >>> (Integer.SIZE - distance));
	}
	
	/**
	 * @accelerable
	 */
	public static int rotateRight(int i, int distance) {
		return (i >>> distance) | (i << (Integer.SIZE - distance));
	}
	
	/**
	 * @accelerable
	 */
	public static int signum(int i) {
		if(i == 0) {
			return 0;
		}
		return (i >> (Integer.SIZE - 1)) | 1;
	}

	public static String toBinaryString(int i) {
		return SupportNumber.toUnsignedString(i, 1);
	}

	public static String toHexString(int i) {
		return SupportNumber.toUnsignedString(i, 4);
	}

	public static String toOctalString(int i) {
		return SupportNumber.toUnsignedString(i, 3);
	}

	public static String toString(int i) {
		return Integer.toString(i, 10);
	}

	public static String toString(int i, int radix) {
		return SupportNumber.toStringInteger(i, radix);
	}

	public static Integer valueOf(int value) {
		// Integer Instantiation : Avoid instantiating Integer objects. Call Integer.valueOf() instead.
		// we are in valueOf impl.
		return new Integer(value); //NOSONAR
	}

	public static Integer valueOf(String s) throws NumberFormatException {
		return valueOf(s, 10);
	}

	public static Integer valueOf(String s, int radix) throws NumberFormatException {
		// Integer Instantiation : Avoid instantiating Integer objects. Call Integer.valueOf() instead.
		// we are in valueOf impl.
		return new Integer(Integer.parseInt(s, radix));  //NOSONAR
	}

	/* INSTANCE VARIABLES */
	private int value;

	public Integer(int value) {
		this.value = value;
	}
	
	public Integer(String s) throws NumberFormatException {
		this(valueOf(s).intValue());
	}

	public byte byteValue() {
		return (byte) this.value;
	}

	public int compareTo(Integer o) {
		return compare(this.value, o.value);
	}

	public double doubleValue() {
		return (double) this.value;
	}

	public boolean equals(@Nullable Object obj) {
		try {
			return ((Integer) obj).value == this.value;
		} catch (ClassCastException e) {
			// obj is not an Integer
			return false;
		} catch (NullPointerException e) {
			// obj is null
			return false;
		}
	}

	public float floatValue() {
		return (float) this.value;
	}

	public int hashCode() {
		return this.value;
	}

	public int intValue() {
		return this.value;
	}

	public long longValue() {
		return (long) this.value;
	}

	public short shortValue() {
		return (short) this.value;
	}

	public String toString() {
		return Integer.toString(this.value, 10);
	}

}
