/*
 * Java
 *
 * Copyright 2008-2019 IS2T. All rights reserved. 
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.lang;

import com.is2t.vm.support.lang.SupportNumber;

import ej.annotation.Nullable;

public final class Float extends Number implements Comparable<Float> {


	public static final int MAX_EXPONENT = SupportNumber.FLOAT_MAX_EXPONENT;

	public static final float MAX_VALUE = SupportNumber.FLOAT_MAX_VALUE;

	public static final int MIN_EXPONENT = SupportNumber.FLOAT_MIN_EXPONENT;

	public static final float MIN_VALUE = SupportNumber.FLOAT_MIN_VALUE;

	public static final float NaN = SupportNumber.FLOAT_NaN;

	public static final float NEGATIVE_INFINITY = SupportNumber.FLOAT_NEGATIVE_INFINITY;

	public static final float POSITIVE_INFINITY = SupportNumber.FLOAT_POSITIVE_INFINITY;

	public static final int SIZE = SupportNumber.FLOAT_SIZE;

	public static final float MIN_NORMAL = SupportNumber.FLOAT_MIN_NORMAL;

	private static int SIGN_MASK = 0b10000000_00000000_00000000_00000000;

	public static int compare(float f1, float f2) {
		if (Float.isNaN(f1) && Float.isNaN(f2)) {
			return 0;
		} else if (Float.isNaN(f1)) {
			return 1;
		} else if (Float.isNaN(f2)) {
			return -1;
		} else {
			boolean d1Positive = (floatToIntBits(f1) & SIGN_MASK) == 0;
			boolean d2Positive = (floatToIntBits(f2) & SIGN_MASK) == 0;

			if (d1Positive && d2Positive || !d1Positive && !d2Positive) {
				return f1 > f2 ? 1 : f1 < f2 ? -1 : 0;
			} else if (d1Positive) {
				return 1;
			} else {
				return -1;
			}
		}
	}

	public static native int floatToIntBits(float value);

	native public static int floatToRawIntBits(float value);

	public static native float intBitsToFloat(int bits);

	public static boolean isInfinite(float v) {
		return v == Float.POSITIVE_INFINITY || v == Float.NEGATIVE_INFINITY;
	}

	public static boolean isNaN(float v) {
		return v != v;
	}

	public static float parseFloat(String s) throws NumberFormatException {
		return SupportNumber.parseFloat(s);
	}

//	public static String toHexString(float f) {
//	}

	public static String toString(float f) {
		return SupportNumber.toStringFloat(f);
	}

	public static Float valueOf(float value) {
		return new Float(value);
	}

	public static Float valueOf(String s) throws NumberFormatException {
		return new Float(Float.parseFloat(s));
	}

	/* INSTANCE VARIABLES */
	private final float value;

	public Float(double value) {
		this.value = (float) value;
	}

	public Float(float value) {
		this.value = value;
	}

	public Float(String s) throws NumberFormatException {
		this(valueOf(s).floatValue());
	}

	@Override
	public byte byteValue() {
		return (byte) this.value;
	}

	@Override
	public int compareTo(Float o) {
		return compare(this.value, o.value);
	}

	@Override
	public double doubleValue() {
		return this.value;
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		try {
			return compare(value, ((Float) obj).value) == 0;
		} catch (NullPointerException | ClassCastException e) {
			return false;
		}
	}

	@Override
	public float floatValue() {
		return this.value;
	}

	@Override
	public int hashCode() {
		return Float.floatToIntBits(this.value);
	}

	@Override
	public int intValue() {
		return (int) this.value;
	}

	public boolean isInfinite() {
		return Float.isInfinite(this.value);
	}

	public boolean isNaN() {
		return Float.isNaN(this.value);
	}

	@Override
	public long longValue() {
		return (long) this.value;
	}

	@Override
	public short shortValue() {
		return (short) this.value;
	}

	@Override
	public String toString() {
		return Float.toString(this.value);
	}
}
