/*
 * Java
 *
 * Copyright 2004-2021 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.lang;

import java.io.ByteArrayInputStream;
import java.io.InputStream;

import com.is2t.s3.io.ResourceInputStream;
import com.is2t.vm.support.lang.Systools;

import ej.annotation.Nullable;

/** IST-API
 * @author IST
 * 
 * Instances of the class Class represent classes and interfaces in a running Java application.
 * Every array also belongs to a class that is reflected as a Class object that is shared by all
 * arrays with the same element type and number of dimensions.
 * 
 * Class has no public constructor. Instead Class objects are constructed automatically by the
 * Java Virtual Machine as classes are loaded.
 * 
 * The following example uses a Class object to print the class name of an object:
 * 
 *           void printClassName(Object obj) {
 *               System.out.println("The class of " + obj +
 *                                  " is " + obj.getClass().getName());
 *           }
 */
public final class Class<T> implements java.lang.reflect.AnnotatedElement, java.lang.reflect.GenericDeclaration, java.io.Serializable, java.lang.reflect.Type {
	
	private static final String EDC_PROPERTIES_PREFIX = "com.is2t.edc.";
	
	private static final String ASSERTIONS_STATUS_PROPERTY = "assertions.status" ;

	
	//the vm remembers the underlying object that correspond to the class
	//in the native world. *this field is set by Class.forName(...)*
	private int internalPtr;
    
    /**  IST-API
     */
    private Class()
    {
        //cannot be instanciated
    }
    
    @SuppressWarnings("unchecked")
	public <U> Class<? extends U> asSubclass(Class<U> clazz) {
		// Check if cast is possible instead of trying to cast and catch exception
    	// since isAssignableFrom is native
    	if(clazz.isAssignableFrom(this)) {
    		return (Class<? extends U>) this;
    	}
    	else {
    		throw new ClassCastException();
    	}
	}
    
    @SuppressWarnings("unchecked")
	@Nullable
    public T cast(@Nullable Object obj) {
    	// See javadoc :
    	// throws ClassCastException if the object is not null and is not
    	// assignable to the type T.
		if(obj != null && !this.isInstance(obj)) {
			throw new ClassCastException();
		}
		
		return (T) obj;
	}
    
    public boolean desiredAssertionStatus() {
        return new Boolean(System.getPropertyNoCheck(new StringBuilder(EDC_PROPERTIES_PREFIX).append(ASSERTIONS_STATUS_PROPERTY).toString()));
    }
    
//  public Class<?>[] getInterfaces() {
//	}
  
    /**
     * Returns the {@code Class} representing the superclass of the entity
     * (class, interface, primitive type or void) represented by this
     * {@code Class}.  If this {@code Class} represents either the
     * {@code Object} class, an interface, a primitive type, or void, then
     * null is returned.  If this object represents an array class then the
     * {@code Class} object representing the {@code Object} class is
     * returned.
     *
     * @return the superclass of the class represented by this object.
     */
    @Nullable
    native public Class<? super T> getSuperclass();
	
	/**  IST-API
	 * Returns the Class object associated with the class with the given string name.
	 * Given the fully-qualified name for a class or interface, this method attempts to locate,
	 * load and link the class.
	 * 
	 * For example, the following code fragment returns the runtime Class descriptor for the
	 * class named java.lang.Thread:
	 * 
     *         Class t = Class.forName("java.lang.Thread")
	 */
	native public static Class<?> forName(String className)
		throws ClassNotFoundException ;

    
//    static Class forName(String className,boolean initialize,ClassLoader loader) throws ClassNotFoundException{
//        return forName(className);
//    }
    
    /**  IST-API
     * Returns the fully-qualified name of the entity (class, interface, array class, primitive type, or void)
     * represented by this Class object, as a String.
     * 
     * If this Class object represents a class of arrays, then the internal form of the name consists of
     * the name of the element type in Java signature format, preceded by one or more "[" characters
     * representing the depth of array nesting. Thus:
     * 
     *             (new Object[3]).getClass().getName()
     * 
     *  returns "[Ljava.lang.Object;" and:
     * 
     *            (new int[3][4][5][6][7][8][9]).getClass().getName()
     * 
     *  returns "[[[[[[[I". The encoding of element type names is as follows:
     *
     *       B            byte
     *       C            char
     *       D            double
     *       F            float
     *       I            int
     *       J            long
     *       Lclassname;  class or interface
     *       S            short
     *       Z            boolean
     * 
     * The class or interface name classname is given in fully qualified form as shown in the example above.
     */
	native public String getName() ;
	
	
	/**
	 * Gets the package for this class. Null is returned if no package object was created by the
	 * class loader of this class.
	 * 
	 * @return the package of the class, or null if no package information is available from the
	 *         archive or codebase.
	 */
	@Nullable
	public Package getPackage() {
		return Package.getPackage(this);
	}
	
	/**
	 * Returns the simple name of the underlying class as given in the source code. Returns an empty
	 * string if the underlying class is anonymous.
	 * 
	 * <p>
	 * The simple name of an array is the simple name of the component type with "[]" appended. In
	 * particular the simple name of an array whose component type is anonymous is "[]".
	 * 
	 * @return the simple name of the underlying class
	 */
	public String getSimpleName() {
		return Systools.getSimpleName(getName());
	}

	
	/**  IST-API
	 * Finds a resource with a given name in the application's JAR file. This
	 * method returns null if no resource with this name is found in the
	 * application's JAR file.
	 * 
	 * The resource names can be represented in two different formats: absolute or
	 * relative.
	 * 
	 * Absolute format: /packagePathName/resourceName
	 * 
	 * Relative format: resourceName
	 * 
	 * In the absolute format, the programmer provides a fully qualified name that
	 * includes both the full path and the name of the resource inside the JAR
	 * file. In the path names, the character "/" is used as the separator.
	 * 
	 * In the relative format, the programmer provides only the name of the actual
	 * resource. Relative names are converted to absolute names by the system by
	 * prepending the resource name with the fully qualified package name of class
	 * upon which the getResourceAsStream method was called.
	 */
	@Nullable
	public InputStream getResourceAsStream(String name)
    {
		if (name.length() == 0)
			return null;
		
        if(name.charAt(0)==(byte)'/')//absolute path
            name = name.substring(1);
        else
        {   //relative path
            String classPackage = getName().replace('.', '/');//get the package of the current class
            int lastIndex = classPackage.lastIndexOf('/');
            if (lastIndex != -1){	//package name can be null! (default package)
                 name = new StringBuilder().append(classPackage.chars,0,lastIndex+1).append(name).toString();
            }
        }
        
        int handle = getResourceHandle(name.getBytes());
        if (handle != 0) {
        	return new ResourceInputStream(handle);
        } else {
        	return null;
        }
    }

	native private int getResourceHandle(byte[] name);
	
	@Nullable
    native private byte[] getResourceData(byte[] name);
    
	
	/**  IST-API
	 * Determines if this Class object represents an array class.
	 */
    native public boolean isArray() ;
   
  
	/** IST-API
	 * Determines if the class or interface represented by this Class object is
	 * either the same as, or is a superclass or superinterface of, the class
	 * or interface represented by the specified Class parameter. It returns
	 * true if so; otherwise it returns false. If this Class object represents
	 * a primitive type, this method returns true if the specified Class
	 * parameter is exactly this Class object; otherwise it returns false.
	 * 
	 * Specifically, this method tests whether the type represented by the
	 * specified Class parameter can be converted to the type represented by
	 * this Class object via an identity conversion or via a widening reference
	 * conversion. See The Java Language Specification, sections 5.1.1 and
	 * 5.1.4 , for details.
	 */
	native public boolean isAssignableFrom(Class<?> cls) ;

	/** IST-API
	 * Determines if the specified Object is assignment-compatible with the
	 * object represented by this Class. This method is the dynamic equivalent
	 * of the Java language instanceof operator. The method returns true if the
	 * specified Object argument is non-null and can be cast to the reference
	 * type represented by this Class object without raising a
	 * ClassCastException. It returns false otherwise.
	 * 
	 * Specifically, if this Class object represents a declared class, this
	 * method returns true if the specified Object argument is an instance of
	 * the represented class (or of any of its subclasses); it returns false
	 * otherwise. If this Class object represents an array class, this method
	 * returns true if the specified Object argument can be converted to an
	 * object of the array class by an identity conversion or by a widening
	 * reference conversion; it returns false otherwise. If this Class object
	 * represents an interface, this method returns true if the class or any
	 * superclass of the specified Object argument implements this interface;
	 * it returns false otherwise. If this Class object represents a primitive
	 * type, this method returns false.
	 */
	native public boolean isInstance(@Nullable Object obj) ;
   
	/**  IST-API
	 * Determines if the specified Class object represents an interface type.
	 */
    native public boolean isInterface() ;

	/**  IST-API
	 * Creates a new instance of a class.
	 */
    native public Object newInstance()
		throws InstantiationException, IllegalAccessException ;
   
	/** IST-API
	 * Converts the object to a string. The string representation is the string
	 * "class" or "interface", followed by a space, and then by the fully
	 * qualified name of the class in the format returned by getName. If this
	 * Class object represents a primitive type, this method returns the name
	 * of the primitive type. If this Class object represents void this method
	 * returns "void".
	 */
	@Override
	public String toString() {
		//WARNING It's impossible to invoke toString() for Class objects which represents a primitive type
		//because the field TYPE does not exist inside wrapper classes in CLDC 1.1(Integer, Double ...)
		//i.e. you can't write : int.class.toString()
		
	/*	if (this == Integer.TYPE ||
			this == Short.TYPE ||
			this == Byte.TYPE ||
			this == Long.TYPE ||
			this == Double.TYPE ||
			this == Char.TYPE ||
			this == Float.TYPE ||
			this == Void.TYPE)
		return getName(); */
	
		if (isInterface())
			return "interface " + getName();
		else
			return "class " + getName();
	}
    
    private static Class<?> forName(String name, boolean bool, ClassLoader classloader) throws ClassNotFoundException
    {   //this method is only used when debugging on the RPE
        return forName(name);
    }
   
}


class ClassLoader //only for debug
{
    
}