/*
 * Java
 * 
 * Copyright 2008-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.lang;

import ej.annotation.Nullable;

public final class Character implements java.io.Serializable, Comparable<Character> {
	
	/* CLASS CONSTANTS */
	public static final int MIN_RADIX = 2 ;
	public static final int MAX_RADIX = 36 ;
	public static final char MIN_VALUE = '\u0000';
	public static final char MAX_VALUE = '\uFFFF';
	public static final int SIZE = 16;
	
	private static final char NULL_CHAR = '\0';
	
	private static final int[] STANDARD_UNICODE_SPACES = {
		'\u0020', '\u00A0', '\u1680', '\u180E', '\u2000', '\u2001', '\u2002', 
		'\u2003', '\u2004', '\u2005', '\u2006', '\u2007', '\u2008', '\u2009', 
		'\u200A', '\u202F', '\u205F', '\u3000'
	};
	
	private static final int[] STANDARD_WHITESPACES = {
		'\u005Ct', '\u005Cn', '\u000B', '\u005Cf', '\u005Cr', '\u001C', '\u001D',
		'\u001D', '\u001E', '\u001F'
	};

	
	/* INSTANCE VARIABLES */
	private char value ;
	
	/**
	 * @accelerable
	 */
	public static int compare(char x, char y) {
		return x - y;
	}
	
	public static int digit(byte ch, int radix) {
        return digit((char)(ch&0xFF), radix);
    }
	
	/**
	 * @accelerable
	 */
	public static int digit(char ch, int radix) {
		int digit = -1 ;
		if(ch >= '0' && ch <= '9') {
			digit = ch - '0' ;
		}
		else if(ch >= 'a' && ch <= 'z') {
			digit = ch - 'a' + 10 ;
		}
		else if(ch >= 'A' && ch <= 'Z') {
			digit = ch - 'A' + 10 ;
		}
		else {
			return -1 ; // arbitrary value (no specs, sun jvm returns -1)
		}
		
		// radix must be 2 through 36 and the digit must be 0 through `radix-1`
		if(radix >= Character.MIN_RADIX 
				&& radix <= Character.MAX_RADIX 
				&& digit < radix) {
			return digit ;
		}
		
		return -1 ;  // arbitrary value (no specs, sun jvm returns -1)
	}
	
	/**
	 * @accelerable
	 */
	public static char forDigit(int digit, int radix) {
		//The digit  argument is valid if 0 <= digit < radix.
		if (digit >= radix || digit < 0) {
            return NULL_CHAR;
        }
		// The radix argument is valid if it is greater than or equal 
		// to MIN_RADIX and less than or equal to MAX_RADIX. 
        if (radix < Character.MIN_RADIX || radix > Character.MAX_RADIX) {
            return NULL_CHAR;
        }
        // If the digit is less than 10, then '0' + digit is returned. 
        if (digit < 10) {
            return (char) ('0' + digit);
        }
        
        // Otherwise, the value 'a' + digit - 10 is returned.
        return (char) ('a' - 10 + digit);
	}
	
	/**
	 * @accelerable
	 */
	public static boolean isDigit(char ch) {
		// TODO 8381
		return ch >= '0' && ch <= '9';
	}
	
	/**
	 * @accelerable
	 */
	public static boolean isISOControl(char ch) {
		return (ch >= 0x0 && ch <= 0x1F) || (ch >= 0x7F && ch <= 0x9F);
	}
	
	/**
	 * @accelerable
	 */
	public static boolean isLowerCase(char ch) {
		// NOTE: this method has been inlined in MicroJvm native side. Report any changes.
		return (ch >= 'a' && ch <= 'z') || (ch >= '\u00DF' && ch <= '\u00FF' && ch != '\u00F7') ;
	}
	
	/**
	 * @accelerable
	 */
	public static boolean isSpaceChar(char ch) {
		for(int i = 0; i < STANDARD_UNICODE_SPACES.length; i++) {
			if(ch == STANDARD_UNICODE_SPACES[i]) {
				return true;
			}
		}
		return false;
	}
	
	/**
	 * @accelerable
	 */
	public static boolean isUpperCase(char ch) {
		// NOTE: this method has been inlined in MicroJvm native side. Report any changes.
		return (ch >= 'A' && ch <= 'Z') || (ch >= '\u00C0' && ch <= '\u00DE' && ch != '\u00D7') ;
	}
	
	/**
	 * @accelerable
	 */
	public static boolean isWhitespace(char ch) {
		if(ch == '\u00A0' || ch == '\u2007' || ch == '\u202F') {
			return false;
		}
		
		return isSpaceChar(ch) || isStandardWhiteSpace(ch);
	}
	
	private static boolean isStandardWhiteSpace(char ch) {
		for(int i = 0; i < STANDARD_WHITESPACES.length; i++) {
			if(ch == STANDARD_WHITESPACES[i]) {
				return true;
			}
		}
		return false;
	}
	
	// TODO WI 8382
//	public static char reverseBytes(char ch) {
//	}
	
	/**
	 * @accelerable
	 */
	public static char toLowerCase(char ch) {
		// NOTE: this method has been inlined in MicroJvm native side. Report any changes.
		// all upper cases have an equivalent lower case
		if(isUpperCase(ch)) {
			return (char) (ch + 0x20);
		}
		
		return ch;
	}
	
	public static String toString(char c) {
		return String.valueOf(c);
	}
	
	/**
	 * @accelerable
	 */
	public static char toUpperCase(char ch) {
		// NOTE: this method has been inlined in MicroJvm native side. Report any changes.
		// all lower cases have an equivalent upper case except \u00DF and \u00FF
		if(isLowerCase(ch) && (ch != '\u00DF') && (ch != '\u00FF')) {
			return (char) (ch - 0x20);
		}
		
		return ch;
	}
	
	public static Character valueOf(char value) {
		return new Character(value) ;
	}
	
	public Character(char value) {
		this.value = value ;
	}
	
	public char charValue() {
		return this.value ;
	}
	
	public int compareTo(Character anotherCharacter) {
		return compare(this.value, anotherCharacter.value);
	}
	
	public boolean equals(@Nullable Object obj) {
		try {
			return ((Character)obj).value == this.value ;
		}
		catch (ClassCastException | NullPointerException e) {
			return false ;
		}
	}
	
	public int hashCode() {
		return (int) this.value ;
	}

	public String toString() {
		char[] c = new char[]{this.value} ;
		return new String(c) ;
	}
	
}
