/*
 * Java
 *
 * Copyright 2013-2019 IS2T. All rights reserved
 * IS2T PROPRIETARY/CONFIDENTIAL Use is subject to license terms.
 */
package java.lang;

import java.io.Serializable;

import com.is2t.tools.ArrayTools;
import com.is2t.vm.support.err.EDCErrorMessages;

import ej.annotation.Nullable;
import ej.error.Message;

public abstract class AbstractStringBuilder implements Appendable, CharSequence, Serializable {

	private static final int DEFAULT_CAPACITY = 16;

	public char[] sharedCharArray;
	public int length;

	public AbstractStringBuilder() {
		this.sharedCharArray = new char[DEFAULT_CAPACITY];
	}

	public AbstractStringBuilder(int capacity) {
		if (capacity < 0) {
			throw new NegativeArraySizeException(String.valueOf(capacity));
		}
		this.sharedCharArray = new char[capacity];
	}

	public void _append(char[] str, int offset, int len) {
		ArrayTools.checkBounds(str.length, offset, len);
		int length = this.length;
		ensureCapacity(length + len);
		System.arraycopy(str, offset, this.sharedCharArray, length, len);
		this.length = length + len;
	}

	public void _append(@Nullable CharSequence s) {
		// Do not call super method since it may call unsynchronized methods
		if(s == null) {
			s = "null"; // see javadoc //$NON-NLS-1$
		}
		// For speed reason, try to guess the real class since for String and StringBuffer
		// the impl. uses System.arraycopy (faster than copying all chars of the CharSequence
		// one by one)
		if(s instanceof AbstractStringBuilder) {
			_append((AbstractStringBuilder) s);
			return;
		}
		else if(s instanceof String) {
			_append((String) s);
			return;
		}
		else {
			_append(s, 0, s.length());
			return;
		}
	}
	
	public void _append(@Nullable AbstractStringBuilder sb){
		if (sb == null) {
			_append("null"); //$NON-NLS-1$
			return;
		}
		_append(sb.sharedCharArray, 0, sb.length);
	}

	public void _append(@Nullable CharSequence csq, int start, int end) {
		if (csq == null) {
			csq = "null"; //$NON-NLS-1$
			start = 0;
			end = 4;
		}

		if (start < 0 || start > end) {
			throw getOutOfRangeMessage(EDCErrorMessages.StringBuilderStartOutOfRange, start);
		}
		if (end > csq.length()) {
			throw getOutOfRangeMessage(EDCErrorMessages.StringBuilderEndOutOfRange, end);
		}
		
		int length = this.length;
		ensureCapacity(length + (end - start));
		char[] sharedCharArray = this.sharedCharArray;
		for (int i = start; i < end; i++) {
			sharedCharArray[length++] = csq.charAt(i);
		}
		this.length = length;
	}
	

	private IndexOutOfBoundsException getOutOfRangeMessage(int id, int value) {
		return new StringIndexOutOfBoundsException(Message.at(new EDCErrorMessages(), id, new Object[]{Integer.valueOf(value)}));
	}

	public void _append(@Nullable String str) {
		if (str == null) {
			str = "null"; //$NON-NLS-1$
		}
		_append(str.chars, str.offset, str.length());
	}

	public int capacity() {
		return this.sharedCharArray.length;
	}

	public char charAt(int index) {
		if (index < 0 || index >= length) {
			throw new IndexOutOfBoundsException(String.valueOf(index));
		}
		return this.sharedCharArray[index];
	}

	public void _delete(int start, int end) {
		if (start < 0 || start > this.length) {
			throw getOutOfRangeMessage(EDCErrorMessages.StringBuilderStartOutOfRange, start);
		}
		end = Math.min(end, length);
		if (start > end) {
			throw getOutOfRangeMessage(EDCErrorMessages.StringBuilderStartOutOfRange, start);
		}
		if (start == end) {
			// Nothing to do
			return;
		}

		// Side effect: resize sequence to it minimal size
		char[] sharedCharArray = this.sharedCharArray;
		int length = this.length;
		System.arraycopy(sharedCharArray, end, sharedCharArray, start, length - end);
		this.length = length - (end - start);
	}

	public void _deleteCharAt(int index) {
		// Can't be done by delete(int, int) because if you want to remove the last char it will not
		// work
		if (index < 0 || index >= this.length) {
			throw new StringIndexOutOfBoundsException(index);
		}
		char[] sharedCharArray = this.sharedCharArray;
		int length = this.length;
		System.arraycopy(sharedCharArray, index + 1, sharedCharArray, index, length - index - 1);
		this.length = length - 1;

	}

	public void ensureCapacity(int minimumCapacity) {
		if (minimumCapacity < 0 || (minimumCapacity <= this.sharedCharArray.length)) {
			return;
		}

		// resize
		char[] sharedCharArray = this.sharedCharArray;
		int newCapacity = Math.max((sharedCharArray.length * 2) + 2, minimumCapacity);
		System.arraycopy(sharedCharArray, 0, this.sharedCharArray = new char[newCapacity], 0, this.length);
	}

	public void getChars(int srcBegin, int srcEnd, char[] dst, int dstBegin) {
		if (srcEnd > this.length) {
			throw new IndexOutOfBoundsException(String.valueOf(srcEnd));
		}
		System.arraycopy(this.sharedCharArray, srcBegin, dst, dstBegin, srcEnd - srcBegin);
	}

	public int indexOf(String str, int fromIndex) {
		// FIXME need a faster way than instantiate a String
		return (new String(this.sharedCharArray, 0, this.length).indexOf(str, fromIndex));
	}

	public void _insert(int index, char[] str, int offset, int len) {
		if (index == this.length) {
			// Insert at end --> append
			_append(str, offset, len);
			return;
		}

		if (index < 0 || index > this.length) {
			throw new StringIndexOutOfBoundsException();
		}
		ArrayTools.checkStringBounds(str.length, offset, len);
		
		int length = this.length;
		ensureCapacity(length + len);
		char[] sharedCharArray = this.sharedCharArray;
		System.arraycopy(sharedCharArray, index, sharedCharArray, index + len, length - index);
		System.arraycopy(str, offset, sharedCharArray, index, len);

		this.length = length + len;

	}

	public void _insert(int dstOffset, @Nullable CharSequence s) {
		if(s == null) {
			s = "null"; //$NON-NLS-1$
		}
		// same as StringBuffer.append(CharSequence)
		if(s instanceof String) {
			_insert(dstOffset, (String) s);
			return;
		}
		
		_insert(dstOffset, s, 0, s.length());
		return;
	}

	public void _insert(int dstOffset, @Nullable CharSequence s, int start, int end) {
		if (dstOffset == this.length) {
			// Insert at end --> append
			_append(s, start, end);
			return;
		}
		
		if(s == null) {
			s = "null"; //$NON-NLS-1$
		}

		if (dstOffset < 0 || dstOffset > this.length) {
			throw new StringIndexOutOfBoundsException();
		}
		if (start < 0 || start > end || end < 0 || end > s.length()) {
			throw new StringIndexOutOfBoundsException();
		}

		int len = end - start;
		int length = this.length;
		ensureCapacity(length + len);
		char[] sharedCharArray = this.sharedCharArray;
		System.arraycopy(sharedCharArray, dstOffset, sharedCharArray, dstOffset + len, length - dstOffset);
		for (int i = start; i < end; i++) {
			sharedCharArray[dstOffset + i] = s.charAt(i);
		}

		this.length = length + len;
	}


	public void _insert(int offset, @Nullable String str) {
		if(str == null) {
			str = "null"; //$NON-NLS-1$
		}
		_insert(offset, str.chars, str.offset, str.length);
	}

	public int lastIndexOf(String str) {
		return lastIndexOf(str, this.length);
	}

	public int lastIndexOf(String str, int fromIndex) {
		//FIXME
		return toString().lastIndexOf(str, fromIndex);
	}

	public int length() {
		return this.length;
	}

	public void _replace(int start, int end, String str) {
		if (start < 0 || start > end || start > this.length) {
			throw new StringIndexOutOfBoundsException();
		}

		int removedLen = end - start;
		int length = this.length;
		int strLength = str.length;
		ensureCapacity(length + strLength - removedLen);
		char[] sharedCharArray = this.sharedCharArray;
		System.arraycopy(sharedCharArray, end, sharedCharArray, start + strLength, length - end);
		System.arraycopy(str.chars, str.offset, sharedCharArray, start, strLength);

		this.length = length - removedLen + strLength;

	}

	public void _reverse() {
		char tmp;
		int j;
		int length = this.length;
		char[] sharedCharArray = this.sharedCharArray;
		int flipCount = length / 2;
		for (int i = 0; i < flipCount; i++) {
			j = length - i - 1;
			tmp = sharedCharArray[i];
			sharedCharArray[i] = sharedCharArray[j];
			sharedCharArray[j] = tmp;
		}
	}

	public void setCharAt(int index, char ch) {
		if (index < 0 || index >= this.length) {
			throw new StringIndexOutOfBoundsException(index);
		}
		this.sharedCharArray[index] = ch;
	}

	public void setLength(int newLength) {
		if (newLength < 0) {
			throw new StringIndexOutOfBoundsException(newLength);
		}

		ensureCapacity(newLength);
		int length = this.length;
		char[] sharedCharArray = this.sharedCharArray;
		if (length < newLength) {
			// Fill empty space with \0
			for (int i = length; i < newLength; i++) {
				sharedCharArray[i] = '\0';
			}
		}
		this.length = newLength;
	}

	public CharSequence subSequence(int start, int end) {
		return substring(start, end);
	}

	public String substring(int start) {
		return substring(start, this.length);
	}

	public String substring(int start, int end) {
		if (start < 0 || start > this.length) {
			throw new StringIndexOutOfBoundsException(start);
		}
		if (end < 0 || end > this.length) {
			throw new StringIndexOutOfBoundsException(end);
		}
		if (start > end) {
			throw new StringIndexOutOfBoundsException();
		}

		return new String(this.sharedCharArray, start, end - start);
	}

	@Override
	public String toString() {
		return new String(this.sharedCharArray, 0, this.length);
	}

	public void trimToSize() {
		// Not necessary if length == capacity
		if (this.length < this.sharedCharArray.length) {
			System.arraycopy(this.sharedCharArray, 0, this.sharedCharArray = new char[this.length], 0, this.length);
		}
	}
}
