/*
 * Java
 *
 * Copyright 2008-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.io;

import com.is2t.vm.support.err.EDCErrorMessages;

import ej.error.Message;

public abstract class Reader implements Readable, Closeable {

	/* IS2T CONSTANTS */
	private static final int SKIP_BUFFER_SIZE = 512 ;

    protected Object lock;

    protected Reader() {
    	this.lock = this ;
    }
    
    @SuppressWarnings({ "null", "unused" }) // (EDC testsuite contains some tests with a null argument) 
    protected Reader(Object lock) {
    	if (lock == null) {
			throw new NullPointerException();
		}
    	this.lock = lock ;
    }

    public abstract void close() throws IOException;

    public void mark(int readAheadLimit) throws IOException {
    	throw new IOException(Message.at(new EDCErrorMessages(), EDCErrorMessages.StreamMarkNotSupported));
    }

    public boolean markSupported() {
    	// The default implementation always returns false. 
    	return false ;
    }

    public int read() throws IOException {
    	char[] array = new char[1] ;
    	if(this.read(array, 0, 1)  == -1) {
    		return -1;
    	}
    	return array[0];
    }

    public int read(char[] cbuf) throws IOException {
    	return this.read(cbuf, 0, cbuf.length) ;
    }

    public abstract int read(char[] cbuf, int off, int len) throws IOException;

    public boolean ready() throws IOException {
    	return false ;
    }

    public void reset() throws IOException {
    	// impose from spec CLDC1.1
    	throw new IOException(Message.at(new EDCErrorMessages(), EDCErrorMessages.ReaderResetNotSupported));
    }

    public long skip(long n) throws IOException {
    	if(n < 0) {
    		throw new IllegalArgumentException() ;
    	}
    		
    	long   oldn = n ;
    	char[] buf = new char[SKIP_BUFFER_SIZE] ;
    	int    length = n > SKIP_BUFFER_SIZE ? SKIP_BUFFER_SIZE : (int)n ;
    	int    lengthRead = 0 ;
    	
    	while(n > 0) {
    		lengthRead = read(buf, 0, length);
	    	if(lengthRead == -1) {
	    		break ;
	    	}
	    	n -= lengthRead ;
	    	length = n > SKIP_BUFFER_SIZE ? SKIP_BUFFER_SIZE : (int)n ;
    	}
    	return oldn-n ;
    }
   
}
