/*
 * Java
 *
 * Copyright 2004-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */ 
package java.io;

import com.is2t.vm.support.CalibrationConstants;

import ej.annotation.Nullable;

public class PrintStream extends FilterOutputStream implements Appendable, Closeable {
	// IMPLEMENTATION NOTES
	// - unless it is not specified, PrintStream is internally thread safe (need to follow Sun reference implementation)	
	// Mentioned in Java Fundamental Classes Reference, By Mark Grand and Jonathan Knudsen; 1-56592-241-7, 1152 pages, 1st Edition May 1997, Chapter 11 - The java.io package - PrintStream (http://docstore.mik.ua/orelly/java/fclass/ch11_52.htm)
	
	private static final String NULL = "null"; //$NON-NLS-1$
	
	/* 
	 * Underlying OutputStreamWriter
	 */
	private OutputStreamWriter outWriter ;
	
	/* 
	 * The error marker which is true when an error has occurred 
	 */
	private boolean errorFlag ;
	private boolean autoFlush;
	

    public PrintStream(OutputStream out) {
    	super(out);
    	this.outWriter = new OutputStreamWriter(out) ;
    }
    
    public PrintStream(OutputStream out, boolean autoFlush) {
    	this(out);
		this.autoFlush = autoFlush;
	}
    
    public PrintStream(OutputStream out, boolean autoFlush, String encoding) throws UnsupportedEncodingException {
		super(out);
		this.outWriter = new OutputStreamWriter(out, encoding) ;
		this.autoFlush = autoFlush;
	}
    
    public PrintStream append(char c) {
		this.print(c);
		return this;
	}
    
    public PrintStream append(@Nullable CharSequence csq) {
    	if(csq == null) {
    		this.print(NULL); // see Javadoc
		} 
    	else {
    		this.print(csq.toString());
		}
    	return this;
	}
    
    public PrintStream append(@Nullable CharSequence csq, int start, int end) {
    	// Same as Writer.append(CharSequence, int, int)
    	CharSequence cs = (csq == null ? NULL : csq);
    	CharSequence subSequence = cs.subSequence(start, end);
		this.print(subSequence.toString());
		return this;
	}

    public boolean checkError() {
    	try {
   			outWriter.flush();
    		return this.errorFlag ;
    	}
    	catch(IOException e) {
    		return this.errorFlag = true ;
    	}
    }
    
    protected void clearError() {
    	this.errorFlag = false;
	}

    public void close() {
    	try {
    		this.outWriter.flush() ; 
    		this.outWriter.close() ;
    	}
    	catch (IOException e) {
    		this.errorFlag = true ;
    	}
    }

    public void flush() {
    	try {
    		this.outWriter.flush() ;
    	}
    	catch (IOException e) {
    		this.errorFlag = true ;
    	}
    }

    public void print(boolean b) {
    	this.print(String.valueOf(b));
    }

    public void print(char c) {
    	this.print(new char[]{c}) ;
    }

	public void print(char[] s) {
		try {
			// delegate to OutputStreamWriter.write to use encoding conversion
			this.outWriter.write(s, 0, s.length);
		} catch (IOException e) {
			this.errorFlag = true;
		}
	}

    public void print(double d) {
    	this.print(Double.toString(d)) ;
    }

    public void print(float f) {
    	this.print(Float.toString(f)) ;
    }

    public void print(int i) {
    	this.print(Integer.toString(i)) ;
    }

    public void print(long l) {
    	this.print(Long.toString(l)) ;
    }

    public void print(@Nullable Object obj) {
   		this.print(String.valueOf(obj)) ;
    }

    public void print(@Nullable String s) {
    	if(s == null) {
    		s = NULL;
    	}
    	try{
    		this.outWriter.write(s.chars, s.offset, s.length);
    	}
    	catch(IOException e) {
    		errorFlag = true;
    	}
    }

    public void println() {
    	try {
    		this.print(System.LineSep) ;
    		if(this.autoFlush) {
				OutputStream out = this.outWriter.out;
				assert out != null;
				out.flush();
    		}
		} catch (IOException e) {
			this.errorFlag = true;
		}
    }

    public void println(boolean x) {
    	synchronized(outWriter.out){ // see implementation note
    		this.print(x) ;
    		this.println() ;
    	}
    }

    public void println(char x) {
    	synchronized(outWriter.out){ // see implementation note
    		this.print(x) ;
    		this.println() ;
    	}
    }

    public void println(char[] x) {
    	synchronized(outWriter.out){ // see implementation note
    		this.print(x) ;
    		this.println() ;
    	}
    }

    public void println(double x) {
    	synchronized(outWriter.out){ // see implementation note
    		this.print(x) ;
    		this.println() ;
    	}
    }

    public void println(float x) {
    	synchronized(outWriter.out){ // see implementation note
    		this.print(x) ;
    		this.println() ;
    	}
    }

    public void println(int x) {
    	synchronized(outWriter.out){ // see implementation note
    		this.print(x) ;
    		this.println() ;
    	}
    }

    public void println(long x) {
    	synchronized(outWriter.out){ // see implementation note
    		this.print(x) ;
    		this.println() ;
    	}
    }

    public void println(Object x) {
    	synchronized(outWriter.out){ // see implementation note
    		this.print(x) ;
    		this.println() ;
    	}
    }

    public void println(@Nullable String x) {
    	if(CalibrationConstants.ENABLE_FAST_PRINTLN) {
    		if(x == null) {
    			x = NULL;
    		}
    		try{
    			this.outWriter.writeln(x.chars, x.offset, x.length);
    		}
    		catch(IOException e) {
    			errorFlag = true;
    		}    		
    	}
    	else {
    		synchronized(outWriter.out) { // see implementation note
    			this.print(x) ;
    			this.println() ;
    		}
    	}
    }

    protected void setError() {
    	this.errorFlag = true ;
    }
    
    public void write(byte[] buf, int off, int len) {
    	try {
			// Use underlying OutputStream to print the byte as given
			OutputStream out = this.outWriter.out;
			assert out != null;
			out.write(buf, off, len);
			if(this.autoFlush) {
				out.flush();
			}
		} catch (IOException e) {
			this.errorFlag = true;
		}
	}

	public void write(int b) {
		try {
			// Use underlying OutputStream to print the byte as given
			OutputStream out = this.outWriter.out;
			assert out != null;
			out.write(b);
			if(b == '\n' && this.autoFlush) {
				out.flush();
			}
		} catch (IOException e) {
			this.errorFlag = true;
		}
	}

}
