/*
 * Java
 *
 * Copyright 2008-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.io;

import com.is2t.vm.support.err.EDCErrorMessages;
import com.is2t.vm.support.util.EncodingConversion;

import ej.annotation.Nullable;
import ej.error.Message;

public class OutputStreamWriter extends Writer {

	/*
	 * Underlying output stream
	 */
	@Nullable
	OutputStream out ;
	/*
	 * Chosen Encoding conversion
	 */
	private EncodingConversion encodingConversion;

	public OutputStreamWriter(OutputStream os) {
		this(os, EncodingConversion.DefaultEncoding);
	}

	public OutputStreamWriter(OutputStream os, String enc) throws UnsupportedEncodingException {
		this(os, EncodingConversion.getEncoding(enc));
	}
		
	@SuppressWarnings({ "null", "unused" })
	private OutputStreamWriter(OutputStream os, EncodingConversion enc){
		// Writer() is called using 'this' as lock
		if (os == null) {
			throw new NullPointerException();
		}
		this.out = os;
		this.encodingConversion = enc;
	}

	public void close() throws IOException {
		synchronized (lock) {
			OutputStream out = this.out;
			if(out != null){
				out.close();
				this.out = null;
			}
		}
	}

	public void flush() throws IOException {
		synchronized (lock) {
			OutputStream out = this.out;
			if(out != null) {
				out.flush();
			}
			else {
				throw new IOException(Message.at(new EDCErrorMessages(), EDCErrorMessages.StreamClosed));
			}
		}
	}
	
	@Nullable 
	public String getEncoding() {
		synchronized (lock) {
			if(this.out != null){
				return this.encodingConversion.getEncoding();
			}
			
			return null;
		}
	}

	public void write(char[] cbuf, int off, int len) throws IOException {
		// IMPLEMENTATION NOTES: report any modifications in {@link #writeln(char[], int, int)}
		synchronized (lock) {
			OutputStream out = this.out;
			EncodingConversion encodingConversion = this.encodingConversion;
			if (out != null) {
				//don't have to test if off,len, off+len are out of bounds,
				//an ArrayIndexOutOfBoundsException will be thrown.
				byte[] bytes = new byte[len*encodingConversion.getMaxBytesPerChar()];
				int nbBytes = encodingConversion.encode(cbuf, new int[]{off}, len, bytes, 0, bytes.length);
				out.write(bytes, 0, nbBytes);
			}
			else {
				throw new IOException(Message.at(new EDCErrorMessages(), EDCErrorMessages.StreamClosed));
			}
		}
	}

	/*
	 * Not in CLDC-1.1-API. Please report any modifications in {@link #write(char[], int, int)}
	 */
	public void writeln(char[] cbuf, int off, int len) throws IOException {
		synchronized (lock) {
			OutputStream out = this.out;
			EncodingConversion encodingConversion = this.encodingConversion;
			if (out != null) {
				//don't have to test if off,len, off+len are out of bounds,
				//an ArrayIndexOutOfBoundsException will be thrown.
				byte[] bytes = new byte[len*encodingConversion.getMaxBytesPerChar()+1];
				int nbBytes = encodingConversion.encode(cbuf, new int[]{off}, len, bytes, 0, bytes.length);
				bytes[nbBytes] = '\n';
				out.write(bytes, 0, nbBytes+1);
			}
			else {
				throw new IOException(Message.at(new EDCErrorMessages(), EDCErrorMessages.StreamClosed));
			}
		}
	}

}
