/*
 * Java
 *
 * Copyright 2008-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.io;

import com.is2t.tools.ArrayTools;
import com.is2t.vm.support.CalibrationConstants;
import com.is2t.vm.support.err.EDCErrorMessages;
import com.is2t.vm.support.util.EncodingConversion;

import ej.annotation.Nullable;
import ej.error.Message;

public class InputStreamReader extends Reader {

	/*
	 * Underlying input stream. 
	 * This field is set to null when stream is closed
	 */
	@Nullable 
	private InputStream in;

	private final byte[] workingBuffer;
	private int offsetPendingByte;
	private int nbPendingBytes;

	/*
	 * Chosen Encoding conversion
	 */
	private final EncodingConversion encodingConversion;

	public InputStreamReader(InputStream in) {
		this(in, EncodingConversion.DefaultEncoding);
	}

	public InputStreamReader(InputStream in, String enc) throws UnsupportedEncodingException {
		this(in, EncodingConversion.getEncoding(enc)); //throws UnsupportedEncondingException
	}

	@SuppressWarnings({ "null", "unused" })// (EDC testsuite contains some tests with a null argument) 
	private InputStreamReader(InputStream in, EncodingConversion enc){
		// Reader() is called using 'this' as lock
		if (in == null) {
			throw new NullPointerException();
		}
		this.in = in;
		this.encodingConversion = enc;
		this.workingBuffer = new byte[CalibrationConstants.READER_WRITER_BUFFER_SIZE];
	}

	@Override
	public void close() throws IOException {
		synchronized (lock) {
			InputStream in = this.in;
			if (in != null) {
				in.close();
				this.in = null;
			}
		}
	}
	
	@Nullable 
	public String getEncoding() {
		if(this.in != null) {
			return this.encodingConversion.getEncoding();
		}

		return null;
	}

	@Override
	public void mark(int readAheadLimit) throws IOException {
		synchronized(lock){
			InputStream in = this.in;
			if(in != null) {
				in.mark(readAheadLimit) ;
				this.nbPendingBytes = this.offsetPendingByte = 0;
			}
			else {
				// cause is implementation dependant
				throw new IOException(Message.at(new EDCErrorMessages(), EDCErrorMessages.StreamClosed)); // NOSONAR
			}
		}
	}

	@Override
	public boolean markSupported(){
		InputStream in = this.in;
		if(in != null) {
			return in.markSupported();
		}
		else {
			return false ;
		}
	}

	@Override
	public int read(char[] cbuf, int off, int len) throws IOException {
		ArrayTools.checkBounds( cbuf.length, off, len);
		// like InputStream.read() if nothing to read return 0...
		if(len == 0) {
			return 0;
		}

		synchronized (lock) {
			InputStream in = this.in;
			EncodingConversion encodingConversion = this.encodingConversion;
			if(in != null) {
				// don't have to test if off, len, off+len are out of bounds
				// an ArrayIndexOutOfBoundsException will be thrown.

				if(nbPendingBytes > 0) {
					int[] offsetPendingByteRef = new int[]{ offsetPendingByte };
					int nbDecoded = encodingConversion.decode(workingBuffer, offsetPendingByteRef, nbPendingBytes, cbuf, off, len);
					if(nbDecoded > 0) {
						int newOffset = offsetPendingByteRef[0];
						nbPendingBytes -= (newOffset-offsetPendingByte);
						offsetPendingByte = newOffset;
						return nbDecoded;
					}
					// not enough pending bytes
					if(offsetPendingByte > 0) {
						// pack
						System.arraycopy(workingBuffer, offsetPendingByte, workingBuffer, 0, nbPendingBytes);
						offsetPendingByte = 0;
					}
				}

				// here, no characters decoded
				// read ahead characters
				while(true) { // until at least a character has been decoded
					int nextOffset = offsetPendingByte+nbPendingBytes;
					if(nextOffset == CalibrationConstants.READER_WRITER_BUFFER_SIZE) {
						// reached the right limit of the buffer without having sufficient bytes to decode 1 character
						// => pack
						System.arraycopy(workingBuffer, offsetPendingByte, workingBuffer, 0, nbPendingBytes);
						offsetPendingByte = 0;
						nextOffset = nbPendingBytes;
					}
					int nbRead = in.read(workingBuffer, nextOffset, CalibrationConstants.READER_WRITER_BUFFER_SIZE-nextOffset);
					if(nbRead == -1) {
						// EOF
						return -1;
					}

					nbPendingBytes += nbRead;

					int[] offsetPendingByteRef = new int[]{ offsetPendingByte };
					int nbDecoded = 0;
					nbDecoded = encodingConversion.decode(workingBuffer, offsetPendingByteRef, nbPendingBytes, cbuf, off, len);
					if(nbDecoded > 0) {
						int newOffset = offsetPendingByteRef[0];
						nbPendingBytes -= (newOffset-offsetPendingByte);
						offsetPendingByte = newOffset;
						return nbDecoded;
					}
				}
			}
			else{
				throw new IOException(Message.at(new EDCErrorMessages(), EDCErrorMessages.StreamClosed));
			}
		}
	}

	@Override
	public boolean ready() throws IOException {
		// Follow J2SE-1.3.1 SPEC: An InputStreamReader is ready if its input buffer is not empty,
		// or if bytes are available to be read from the underlying byte stream.
		synchronized(lock) {
			InputStream in = this.in;
			if(in != null) {
				return nbPendingBytes > 0 || in.available() > 0;
			}
			else {
				// cause is implementation dependant
				throw new IOException(Message.at(new EDCErrorMessages(), EDCErrorMessages.StreamClosed)); //NOSONAR
			}
		}
	}

	@Override
	public void reset() throws IOException {
		synchronized (lock) {
			InputStream in = this.in;
			if(in != null) {
				in.reset();
				this.nbPendingBytes = this.offsetPendingByte = 0;
			}
			else{
				// cause is implementation dependant
				throw new IOException(Message.at(new EDCErrorMessages(), EDCErrorMessages.StreamClosed)); // NOSONAR
			}
		}
	}

}
