/*
 * Java
 *
 * Copyright 2008-2019 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.io;

public class DataInputStream extends FilterInputStream implements DataInput {

    public static final String readUTF(DataInput in) throws IOException {
    	return in.readUTF() ;
	}

    public DataInputStream(InputStream in) {
    	super(in);
    }
    
    public final int read(byte[] b) throws IOException {
		InputStream in = this.in;
		assert in != null; 
    	return in.read(b) ;
	}
    
    public final int read(byte[] b, int off, int len) throws IOException {
		InputStream in = this.in;
		assert in != null; 
    	return in.read(b, off, len) ;
	}
    
    public final boolean readBoolean() throws IOException {
		InputStream in = this.in;
		assert in != null; 
    	int i = in.read() ;
    	if(i==-1) {
    		throw new EOFException() ;
    	}
    	return i > 0 ;
	}
    
    public final byte readByte() throws IOException {
		InputStream in = this.in;
		assert in != null; 
     	int i = in.read() ;
    	if(i==-1) {
    		throw new EOFException() ;
    	}
    	// i is include in [0..255]
    	return (byte)i ;
    }
    
    public final char readChar() throws IOException {
		InputStream in = this.in;
		assert in != null; 
    	//     read assume that char1 is a number in [0.255] or -1 if EOF
      	int char1 = in.read() ;
    	int char2 = in.read() ;
    	
    	if((char1 | char2) <0) {
    		throw new EOFException() ;
    	}

        return (char) ((char1 << 8) | char2);
    }
    
    public final double readDouble() throws IOException {
    	return Double.longBitsToDouble(readLong()) ;
    }

    public final float readFloat() throws IOException {
    	return Float.intBitsToFloat(readInt()) ;
    }

    public final void readFully(byte[] b) throws IOException {
    	this.readFully(b, 0, b.length) ;
    }
    
    public final void readFully(byte[] b, int off, int len) throws IOException {
    	int end = off + len;
    	while(off<end) {
    		int nbRead = read(b, off, end-off);
    		if(nbRead==-1) {
    			throw new EOFException();
    		}
    		off += nbRead;
    	}
    }
    
    public final int readInt() throws IOException {
     	//int a=in.read(), b=in.read(), c=in.read(), d=in.read() ;
    	InputStream in = this.in;
		assert in != null; 
        int ch1 = in.read();
        int ch2 = in.read();
        int ch3 = in.read();
        int ch4 = in.read();
        
        if((ch1 | ch2 | ch3 | ch4) < 0) {
            throw new EOFException();
        }
        
        return (ch1 << 24) + (ch2 << 16) + (ch3 << 8) + ch4;
    }
    
    public final long readLong() throws IOException {
    	long l1 = readInt();
    	long l2 = readInt();
    	return (l1<<32) | (l2&0xFFFFFFFFl);
    }

    public final short readShort() throws IOException {
    	return (short) readChar() ;
    }

    public final int readUnsignedByte() throws IOException {
		InputStream in = this.in;
		assert in != null; 
       	int aByte = in.read();
    	if (aByte < 0) {
    	    throw new EOFException();
    	}
    	
    	return aByte;
    }

    public final int readUnsignedShort() throws IOException {
       	return readChar();
    }
    
    // Sonar : Cyclomatic complexity is fine here
    public final String readUTF() throws IOException { //NOSONAR
		InputStream in = this.in;
		assert in != null; 
		int length = readUnsignedShort();
		char[] reads = new char[length];
		int ptrReads = 0;
		
		while (length > 0) {
			int a = in.read(), b = 0, c = 0;
			if(a == -1) {
				throw new EOFException();
			}
			
			if((a & 0xf0) == 0xf0 || a == 0) {
				throw new UTFDataFormatException();
			}
			
			// decrease length each time a byte is read
			--length;
			
			if((a & 0x80) == 0) { // 0xxxxxxx -> single byte character
				// add the char decoded to the buffer
				reads[ptrReads++]= (char) a;
			}
			else if((a & 0xE0) == 0xE0) { // 1110xxxx -> 3-bytes char
				b = in.read();
				if(b == -1)
					throw new EOFException();
				c = in.read();
				if(c == -1) {
					throw new EOFException();
				}
				if((b & 0xC0) != 0x80 || (c & 0xC0) != 0x80) { // must be 10xxxxxx
					throw new UTFDataFormatException();
				}
				length -= 2;
				if(length < 0) {
					throw new UTFDataFormatException();
				}
				reads[ptrReads++]=  (char) (((a & 0x0F) << 12) | ((b & 0x3F) << 6) | (c & 0x3F));
			}
			else if((a & 0xC0) == 0xC0) { // 110xxxxx -> 2-bytes char
				b = in.read();
				if(b == -1) {
					throw new EOFException();
				}
				if((b & 0xC0) != 0x80) {// must be 10xxxxxx
					throw new UTFDataFormatException();
				}
				--length;
				if(length < 0) {
					throw new UTFDataFormatException();
				}
				reads[ptrReads++]=  (char) (((a & 0x1F) << 6) | (b & 0x3F));
			}
			else {
				throw new UTFDataFormatException();
			}
		}
		return new String(reads, 0, ptrReads);
	}
    
    public final int skipBytes(int n) throws IOException {
		InputStream in = this.in;
		assert in != null; 
    	return (int) in.skip(n) ;
    }

}
