/*
 * Java
 *
 * Copyright 2008-2014 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package java.io;

import com.is2t.tools.ArrayTools;

/* Implementation note:
 * write methods do not throw IOException in CLDC specification
 * On Sun WTK, the behavior is to do nothing: we can write bytes after close. (Tested on Sub WTK 2.5.2)
 * Since J2SE 1.4.2, close() spec has been turned to "close has not effect on the stream"
 * => We follow the WTK behavior
 */
public class ByteArrayOutputStream extends OutputStream {
	
	private static final int DEFAULT_CAPACITY = 32;

    protected byte[] buf;
    protected int count;
    
    public ByteArrayOutputStream() {
    	this.buf = new byte[DEFAULT_CAPACITY] ;
    }

    public ByteArrayOutputStream(int size) {
     	try {
    		this.buf = new byte[size] ;
    	}
     	catch (NegativeArraySizeException e) {
    		 // test specify in cldc1.1
     		// Cause is not needed (implementation specific)
    		throw new IllegalArgumentException(); //NOSONAR
    	}
    }

    @Override
	public void close() throws IOException {
    	// do nothing (see class comment)
    }

    public synchronized void reset() {
    	this.count = 0 ;
    }

    public synchronized int size() {
    	return this.count ;
    }

    public synchronized byte[] toByteArray() {
    	byte[] array = new byte[this.count] ;
    	System.arraycopy(this.buf, 0, array, 0, this.count) ;
    	return array ;
    }

    @Override
	public String toString() {
    	//the conversion byte[]->char[] is done by InputStreamReader
    	//in the future, it will depends of the encoding format
    	StringBuilder sb = new StringBuilder() ;
    	ByteArrayInputStream bis = new ByteArrayInputStream(this.buf, 0, this.count) ;
    	InputStreamReader usr = new InputStreamReader(bis) ;
    	char[] c = new char[this.count] ;
    	try {
    		usr.read(c, 0, c.length) ;
    		sb.append(c) ;
    	}
    	catch (IOException e) { //NOSONAR
    		// fail silently
    	}
    	return sb.toString() ;
    }
    
    @Override
	public synchronized void write(byte[] b, int offset, int len){
        ArrayTools.checkBounds(b, offset, len);
    	
    	//if b==null then arraycopy will throw a NullPointerException (spec)
     	try {
			System.arraycopy(b, offset, this.buf, this.count, len) ;
    	} catch (IndexOutOfBoundsException e ) {
			//  ensureCapacity only if IndexOutOfBoundsException is thrown
	   		ensureCapacity(len) ;
  			System.arraycopy(b, offset, this.buf, this.count, len) ;
    	}
    	this.count += len;
    }

    @Override
	public synchronized void write(int b){
    	try {
	    	this.buf[this.count] = (byte)(b & 0x000000ff)  ;
	    }
    	catch (IndexOutOfBoundsException e) {	// ensureCapacity() only when needed
	    	ensureCapacity(1);
	    	this.buf[this.count] = (byte)(b & 0x000000ff)  ;
    	}
    	count++;
    }
    
    public synchronized void writeTo(OutputStream out) throws IOException {
		out.write(this.buf, 0, this.count);
	}
    
    private void ensureCapacity(int cap) {
    	byte[] buf = this.buf ;
    	int count = this.count ;
    	
    	if (buf.length-count < cap) {
    		byte[] newArray;
    		int newCapacity = Math.max( (buf.length<<1) + 2 , count+cap);
    		System.arraycopy(buf, 0, newArray= new byte[newCapacity], 0, count);
    		this.buf = newArray;
    	}
    }
    
}