/*
 * Java
 *
 * Copyright 2010-2013 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */ 
package com.is2t.vm.support;

import java.util.TimeZone;

import ej.bon.Immutables;
import ej.bon.ImmutablesError;
import ej.error.Message;

/**
 * Implementation framework for {@link TimeZone} provided through BON immutable.
 * The {@link TimeZone} implementation is out of scope.
 * Application must provide an immutable array of {@link TimeZoneEntry} named <code>TIMEZONES</codes>.
 * {@link TimeZoneEntry} are sorted through alphabetical order of {@link TimeZoneEntry#getId()}. 
 * Implementation is focused on small RAM and ROM footprint. Informations are not cached except {@link #Default} timeZone.
 */
public class ImmutableTimeZoneImpl implements TimeZoneImpl{

	/**
	 * The name of the immutable id which provided {@link TimeZone} data.
	 */
	public static final String ARRAY_ID = "TIMEZONES";
	
	private static TimeZone Default;
	
	
	public String[] getAvailableIDs() {
		// array is computed on the fly
		TimeZoneEntry[] entries = getEntries();
		int nbIds = entries.length;
		String[] ids = new String[nbIds];
		for(int i=nbIds; --i>=0;){
			ids[i] = entries[i].getId();
		}
		return ids;
	}

	public TimeZone getDefault() {
		if(Default == null){
			// this is the first time the default TimeZone is needed
			String st = System.getPropertyNoCheck(TimeZone.TimeZoneProperty);
	    	if (st == null)
	    		st = "GMT";
	    	Default = getTimeZone(st);			
		}
		return Default;
	}

	public TimeZone getTimeZone(String id) {
		// dichotomical search within the array
		TimeZoneEntry[] entries = getEntries();
		int step = -1;
		while(true){
			if(++step == 2)
				break;
			
			int left = 0;
			int right = entries.length-1;

			while(left <= right){
				int middle = left+((right-left)/2);
				TimeZoneEntry middleEntry = entries[middle];
				String middleId = middleEntry.getId();
				if(middleId.equals(id)){
					return middleEntry.value;
				}
				if (middleId.compareTo(id) < 0)
					left = middle + 1;
				else
					right = middle - 1;
			}

			// Here, TimeZone is not found. Spec is to fall back to GMT timezone
			// retry with GMT
			id = "GMT";			
		}
		// here, GMT is not found. Fall to default CLDC GMT implementation
		return GMTTimeZone.getInstance();
	}
	
	private static TimeZoneEntry[] getEntries(){
		try{ return (TimeZoneEntry[])Immutables.get(ARRAY_ID); }
		catch(ImmutablesError e){
			throw new RuntimeException(Message.at(new TimeZoneErrorMessages(), TimeZoneErrorMessages.ImmutableNotFound));
		}
		catch(ClassCastException e){
			throw new RuntimeException(Message.at(new TimeZoneErrorMessages(), TimeZoneErrorMessages.InvalidImmutable));
		}
	}

}
