/*
 * Java
 *
 * Copyright 2011-2017 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.is2t.tools;

import ej.error.Message;

///!\ Copied in lib-Tools-j2se, every change must be done in both implementations.
public final class ArrayTools{

	private ArrayTools() {
	}

	//Bounds exception types
	private static final int TYPE_INDEX_OUT_OF_BOUNDS = 1;
	private static final int TYPE_ARRAY_INDEX_OUT_OF_BOUNDS = 2;
	private static final int TYPE_STRING_INDEX_OUT_OF_BOUNDS = 3;

	public static void checkBounds(byte[] b, int off, int len) {
		//if b==null NullPointerException is thrown
		checkBounds(b.length, off, len, TYPE_INDEX_OUT_OF_BOUNDS);
	}

	public static void checkBounds(int arrayLength, int off, int len) {
		checkBounds(arrayLength, off, len, TYPE_INDEX_OUT_OF_BOUNDS);
	}

	public static void checkArrayBounds(int arrayLength, int off, int len) {
		checkBounds(arrayLength, off, len, TYPE_ARRAY_INDEX_OUT_OF_BOUNDS);
	}

	public static void checkStringBounds(int stringLength, int off, int len) {
		checkBounds(stringLength, off, len, TYPE_STRING_INDEX_OUT_OF_BOUNDS);
	}

	private static void checkBounds(int arrayLength, int off, int len, int exceptionType) {
		int offPlusLen = off+len;
		if (off < 0 || len < 0 || offPlusLen > arrayLength || offPlusLen < 0) {
			//now test more precisely for the exception message
			int messages = 0;
			Object arg = "";
			if(off < 0){
				messages=ToolsErrorMessage.ArrayNegativeOffset;
				arg = Integer.toString(off);
			}
			else if(len < 0){
				messages=ToolsErrorMessage.ArrayNegativeLength;
				arg = Integer.toString(len);
			}
			else if(offPlusLen > arrayLength){
				messages=ToolsErrorMessage.ArrayOffsetLengthGreater;
				arg = Integer.toString(off);
			}
			else if(offPlusLen < 0){
				messages=ToolsErrorMessage.ArrayNegativeOffsetPlusLength;
				arg = Integer.toString(offPlusLen);
			}
			else{
				throw new AssertionError();
			}


			String errorMessage = Message.at(new ToolsErrorMessage(),  messages, arg);

			switch(exceptionType){
			case TYPE_ARRAY_INDEX_OUT_OF_BOUNDS:
				throw new ArrayIndexOutOfBoundsException(errorMessage);
			case TYPE_STRING_INDEX_OUT_OF_BOUNDS:
				throw new StringIndexOutOfBoundsException(errorMessage);
			case TYPE_INDEX_OUT_OF_BOUNDS:
			default:
				throw new IndexOutOfBoundsException(errorMessage);
			}
		}
	}

	public static final boolean DEBUG = false;

	public static Object[] add(Object[] array, Object element) {
		if(DEBUG) {
			System.out.println("Add " + element);
		}
		int arrayLength = array.length; // can throw NullPointerException
		Object[] result = (Object[]) createNewArrayFromType(array.getClass(), arrayLength + 1);
		result[arrayLength] = element;
		System.arraycopy(array, 0, result, 0, arrayLength);
		return result;
	}

	public static Object[] add(Object[] array, Object[] other) {
		if(DEBUG) {
			System.out.println("Add " + other);
		}
		int arrayLength = array.length; // can throw NullPointerException
		int otherLength = other.length; // can throw NullPointerException
		Object[] result = (Object[]) createNewArrayFromType(array.getClass(), arrayLength + otherLength);
		System.arraycopy(array, 0, result, 0, arrayLength);
		System.arraycopy(other, 0, result, arrayLength, otherLength);
		return result;
	}

	public static Object[] remove(Object[] array, Object element) {
		if(DEBUG) {
			System.out.println("Remove " + element);
		}
		int arrayLength = array.length;
		for (int i = arrayLength; --i >= 0;) {
			if(array[i] == element) {
				Object[] result = (Object[]) createNewArrayFromType(array.getClass(), arrayLength - 1);
				System.arraycopy(array, 0, result, 0, i);
				System.arraycopy(array, i + 1, result, i, arrayLength - i - 1);
				return result;
			}
		}
		//nothing removed
		return array;
	}

	public static Object[] copy(Object[] array) {
		return copy(array, array.length - 1, array.getClass());
	}

	public static Object[] copy(Object[] array, Class type) {
		return copy(array, array.length - 1, type);
	}

	public static int[] add(int[] array, int element) {
		if(DEBUG) {
			System.out.println("Add " + element);
		}
		int arrayLength = array.length;
		int[] result = new int[arrayLength + 1];
		result[arrayLength] = element;
		System.arraycopy(array, 0, result, 0, arrayLength);
		return result;
	}

	public static int[] remove(int[] array, int element) {
		if(DEBUG) {
			System.out.println("Remove " + element);
		}
		int arrayLength = array.length;
		for (int i = arrayLength; --i >= 0;) {
			if(array[i] == element) {
				int[] result = new int[arrayLength - 1];
				System.arraycopy(array, 0, result, 0, i);
				System.arraycopy(array, i + 1, result, i, arrayLength - i - 1);
				return result;
			}
		}
		//nothing removed
		return array;
	}

	public static int[] removeRange(int[] array, int offset, int length) {
		int arrayLength = array.length;
		int[] newArray = new int[arrayLength+length];
		System.arraycopy(array, 0, newArray, 0, offset);
		System.arraycopy(array, offset+length, newArray, 0, arrayLength-offset-length);
		return newArray;
	}

	public static int[] copy(int[] array) {
		int arrayLength = array.length;
		int[] result = new int[arrayLength];
		System.arraycopy(array, 0, result, 0, arrayLength);
		return result;
	}

	public static Object[] add(Object[] array, Object element, int ptr) {
		if(DEBUG) {
			System.out.println("Add " + element);
		}
		try {
			array[ptr] = element;
		} catch (ArrayIndexOutOfBoundsException e) {
			//grow array
			Object[] result = (Object[]) createNewArrayFromType(array.getClass(), (ptr << 1) + 1);
			System.arraycopy(array, 0, result, 0, array.length);
			array = result;
			array[ptr] = element;
		}
		return array;
	}

	public static boolean remove(Object[] array, Object element, int ptr) {
		if(DEBUG) {
			System.out.println("Remove " + element);
		}
		int arrayLength = array.length;
		for (int i = arrayLength; --i >= 0;) {
			if(array[i] == element) {
				System.arraycopy(array, 0, array, 0, i);
				System.arraycopy(array, i + 1, array, i, ptr - i);
				array[ptr] = null;
				return true;
			}
		}
		//nothing removed
		return false;
	}

	public static Object[] copy(Object[] array, int ptr) {
		return copy(array, ptr, array.getClass());
	}

	public static Object[] copy(Object[] array, int ptr, Class type) {
		int arrayLength = ptr + 1;
		Object[] result = (Object[]) createNewArrayFromType(type, arrayLength);
		System.arraycopy(array, 0, result, 0, arrayLength);
		return result;
	}

	public static int[] add(int[] array, int element, int ptr) {
		if(DEBUG) {
			System.out.println("Add " + element);
		}
		try {
			array[ptr] = element;
		} catch (ArrayIndexOutOfBoundsException e) {
			//grow array
			int[] result = new int[(ptr << 1) + 1];
			System.arraycopy(array, 0, result, 0, array.length);
			array = result;
			array[ptr] = element;
		}
		return array;
	}

	public static boolean remove(int[] array, int element, int ptr) {
		if(DEBUG) {
			System.out.println("Remove " + element);
		}
		int arrayLength = array.length;
		for (int i = arrayLength; --i >= 0;) {
			if(array[i] == element) {
				System.arraycopy(array, 0, array, 0, i);
				System.arraycopy(array, i + 1, array, i, ptr - i);
				array[ptr] = 0;
				return true;
			}
		}
		//nothing removed
		return false;
	}

	public static int[] copy(int[] array, int ptr) {
		int arrayLength = ptr + 1;
		int[] result = new int[arrayLength];
		System.arraycopy(array, 0, result, 0, arrayLength);
		return result;
	}

	//throw NullPointerException if type == null
	//throw ArrayStoreException if type is not an array type
	public static native Object createNewArrayFromType(Class type, int length);

}
